"use client";

import React, { useEffect, useState } from 'react';
import { useTranslation } from 'react-i18next';
import { Card, CardHeader, CardTitle, CardContent } from '../components/ui/card';
import { Button } from '../components/ui/button';
import { Badge } from '../components/ui/badge';
import { useAuth } from '../hooks/use-auth';

interface Order {
  id: number;
  user_id: number;
  f_name: string;
  email: string;
  payment_method: string;
  items: any[];
  total_amount: number;
  status: string;
  created_at: string;
  admin_approved_at?: string;
  user_approved_at?: string;
  document_created_at?: string;
  sumit_document_id?: string;
}

const PendingOrdersTab: React.FC = () => {
  const { t } = useTranslation();
  const { user } = useAuth();
  const [orders, setOrders] = useState<Order[]>([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const fetchOrders = async () => {
    setLoading(true);
    setError(null);
    try {
      const res = await fetch('/api/postgres/orders');
      if (!res.ok) {
        // Если таблица не существует, показываем пустой список
        if (res.status === 500) {
          //console.log('Orders table not found, showing empty list');
          setOrders([]);
          return;
        }
        throw new Error(`HTTP ${res.status}: ${res.statusText}`);
      }
      
      const data = await res.json();
      if (data.orders) {
        setOrders(data.orders);
      } else {
        setOrders([]);
      }
    } catch (err: any) {
      console.error('Error fetching orders:', err);
      // При ошибке показываем пустой список вместо ошибки
      setOrders([]);
    }
    setLoading(false);
  };

  useEffect(() => {
    fetchOrders();
  }, []);

  const handleAdminApprove = async (orderId: number) => {
    try {
      const res = await fetch('/api/postgres/orders', {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          order_id: orderId,
          action: 'admin_approve',
          admin_user_id: (user as any)?.id,
        }),
      });

      const data = await res.json();
      if (data.success) {
        fetchOrders(); // Обновляем список
      } else {
        setError(data.error || 'Ошибка подтверждения заказа');
      }
    } catch (err: any) {
      setError(err.message || 'Ошибка подтверждения заказа');
    }
  };

  const handleUserApprove = async (orderId: number) => {
    try {
      const res = await fetch('/api/postgres/orders', {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          order_id: orderId,
          action: 'user_approve',
        }),
      });

      const data = await res.json();
      if (data.success) {
        fetchOrders(); // Обновляем список
      } else {
        setError(data.error || 'Ошибка подтверждения заказа');
      }
    } catch (err: any) {
      setError(err.message || 'Ошибка подтверждения заказа');
    }
  };

  const getStatusBadge = (status: string) => {
    switch (status) {
      case 'pending':
        return <Badge variant="secondary">{t('orders.pending', 'Ожидает')}</Badge>;
      case 'admin_approved':
        return <Badge variant="outline">{t('orders.adminApproved', 'Подтвержден админом')}</Badge>;
      case 'user_approved':
        return <Badge variant="default">{t('orders.userApproved', 'Подтвержден пользователем')}</Badge>;
      case 'completed':
        return <Badge variant="default">{t('orders.completed', 'Завершен')}</Badge>;
      default:
        return <Badge variant="destructive">{status}</Badge>;
    }
  };

  const getPaymentMethodLabel = (method: string) => {
    switch (method) {
      case 'cash':
        return t('paymentMethods.cash', 'Наличные');
      case 'bank_transfer':
        return t('paymentMethods.bank_transfer', 'Банковский перевод');
      case 'credit_card':
        return t('paymentMethods.credit_card', 'Кредитная карта');
      default:
        return method;
    }
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleString('ru-RU');
  };

  return (
    <Card className="w-full shadow-lg">
      <CardHeader>
        <CardTitle>{t('orders.title', 'Управление заказами')}</CardTitle>
      </CardHeader>
      <CardContent>
        {loading && <div>{t('orders.loading', 'Загрузка заказов...')}</div>}
        {error && <div className="text-red-500 mb-4">{t('orders.error', 'Ошибка загрузки заказов')}: {error}</div>}
        
        <div className="space-y-4">
          {orders.length === 0 ? (
            <div className="text-center text-gray-500 py-8">
              {t('orders.noOrders', 'Нет заказов для отображения')}
            </div>
          ) : (
            orders.map((order) => (
              <Card key={order.id} className="p-4 shadow-lg">
                <div className="flex justify-between items-start mb-4">
                  <div>
                    <h3 className="font-semibold">
                      {t('orders.order', 'Заказ')} #{order.id}
                    </h3>
                    <p className="text-sm text-gray-600">
                      {order.f_name} ({order.email})
                    </p>
                    <p className="text-sm text-gray-600">
                      {formatDate(order.created_at)}
                    </p>
                  </div>
                  <div className="text-right">
                    {getStatusBadge(order.status)}
                    <div className="text-lg font-bold mt-1">
                      ₪{order.total_amount}
                    </div>
                  </div>
                </div>

                <div className="mb-4">
                  <p className="text-sm">
                    <span className="font-medium">{t('orders.paymentMethod', 'Способ оплаты')}:</span>{' '}
                    {getPaymentMethodLabel(order.payment_method)}
                  </p>
                </div>

                <div className="mb-4">
                  <h4 className="font-medium mb-2">{t('orders.items', 'Товары')}:</h4>
                  <div className="space-y-1">
                    {order.items.map((item: any, index: number) => (
                      <div key={index} className="text-sm">
                        {item.name} - {item.quantity} шт. × ₪{item.unit_price} = ₪{item.total_price}
                      </div>
                    ))}
                  </div>
                </div>

                <div className="flex gap-2">
                  {order.status === 'pending' && (
                    <Button
                      onClick={() => handleAdminApprove(order.id)}
                      size="sm"
                      variant="outline"
                    >
                      {t('orders.approveAsAdmin', 'Подтвердить как админ')}
                    </Button>
                  )}
                  
                  {order.status === 'admin_approved' && (
                    <Button
                      onClick={() => handleUserApprove(order.id)}
                      size="sm"
                      variant="default"
                    >
                      {t('orders.approveAsUser', 'Подтвердить как пользователь')}
                    </Button>
                  )}

                  {order.status === 'completed' && (
                    <div className="text-sm text-green-600">
                      {t('orders.documentCreated', 'Документ создан')}
                      {order.sumit_document_id && (
                        <span className="ml-2">(ID: {order.sumit_document_id})</span>
                      )}
                    </div>
                  )}
                </div>

                {order.admin_approved_at && (
                  <div className="text-xs text-gray-500 mt-2">
                    {t('orders.adminApprovedAt', 'Подтвержден админом')}: {formatDate(order.admin_approved_at)}
                  </div>
                )}

                {order.user_approved_at && (
                  <div className="text-xs text-gray-500 mt-2">
                    {t('orders.userApprovedAt', 'Подтвержден пользователем')}: {formatDate(order.user_approved_at)}
                  </div>
                )}

                {order.document_created_at && (
                  <div className="text-xs text-gray-500 mt-2">
                    {t('orders.documentCreatedAt', 'Документ создан')}: {formatDate(order.document_created_at)}
                  </div>
                )}
              </Card>
            ))
          )}
        </div>
      </CardContent>
    </Card>
  );
};

export default PendingOrdersTab; 