import React, { useState, useRef } from 'react';
import { Menu } from 'lucide-react';
import SystemStatsCard from './SystemStatsCard';
import UserSettingsTab from './UserSettingsTab';
import PendingOrdersTab from './PendingOrdersTab';
import PaymentHistoryTab from './PaymentHistoryTab';
import ManagerPushTab from './ManagerPushTab';
import { Button } from '../components/ui/button';
import { useTranslation } from 'react-i18next';

const TABS = [
  { id: 'stats', label: 'managerPanel.stats', component: <SystemStatsCard /> },
  { id: 'user-settings', label: 'managerPanel.userSettings', component: <UserSettingsTab /> },
  { id: 'pending-orders', label: 'managerPanel.pendingOrders', component: <PendingOrdersTab /> },
  { id: 'payment-history', label: 'managerPanel.paymentHistory', component: <PaymentHistoryTab /> },
  { id: 'push', label: 'managerPanel.push', component: <ManagerPushTab /> },
];

const ManagerSystemSettingsTab: React.FC = () => {
  const { t, i18n } = useTranslation();
  const [activeTab, setActiveTab] = useState('stats');
  const [dropdownOpen, setDropdownOpen] = useState(false);
  const dropdownRef = useRef<HTMLDivElement>(null);
  const isMobile = typeof window !== 'undefined' && window.innerWidth <= 768;

  // RTL поддержка
  const dir = i18n.language === 'he' ? 'rtl' : 'ltr';
  const tabListClass = `flex gap-2 mb-4 ${dir === 'rtl' ? 'justify-end' : ''} shadow-lg`;

  // Для RTL: табы справа налево
  const tabsToRender = dir === 'rtl' ? [...TABS].reverse() : TABS;

  return (
    <div dir={dir}>
      {/* Мобильный гамбургер-меню */}
      {isMobile && (
        <div className="flex items-center justify-between px-2 py-2 border-b bg-white sticky top-0 z-40">
          <button onClick={() => setDropdownOpen((v) => !v)} className="p-2 focus:outline-none shadow-lg">
            <Menu className="h-7 w-7" />
          </button>
          
          <div style={{ width: 40 }} />
          {dropdownOpen && (
            <div
              ref={dropdownRef}
              className="absolute z-50 bg-white border rounded shadow-lg flex flex-col animate-fadeIn"
              style={{
                minWidth: 'max-content',
                width: 'max-content',
                maxWidth: '5em',
                [dir === 'rtl' ? 'right' : 'left']: '0.5rem',
                top: '3.5rem',
                direction: dir,
              }}
            >
              {TABS.map(tab => (
                <button
                  key={tab.id}
                  onClick={() => { setActiveTab(tab.id); setDropdownOpen(false); }}
                  className={`flex items-center gap-2 px-4 py-3 text-base transition-colors border-b last:border-b-0 ${activeTab === tab.id ? 'bg-[#549BCE] text-black font-bold' : 'hover:bg-gray-100'}`}
                  style={{ whiteSpace: 'nowrap', overflow: 'hidden', textOverflow: 'ellipsis', direction: dir }}
                >
                  <span>{t(tab.label)}</span>
                </button>
              ))}
            </div>
          )}
        </div>
      )}

      {/* Десктопные табы */}
      {!isMobile && (
        <div
          className={tabListClass + ' justify-center shadow-lg rounded-lg bg-white '}
          style={{
            ...(dir === 'rtl' ? { flexDirection: 'row-reverse' } : {}),
            marginTop: '2.5rem',
            display: 'flex',
            width: '100%',
          }}
        >
          {tabsToRender.map(tab => (
            <Button
              key={tab.id}
              variant={activeTab === tab.id ? 'default' : 'outline'}
              onClick={() => setActiveTab(tab.id)}
              className={activeTab === tab.id ? '!bg-[#549BCE] !text-black font-bold shadow-lg' : ''}
            >
              {t(tab.label)}
            </Button>
          ))}
        </div>
      )}
      <div>{TABS.find(tab => tab.id === activeTab)?.component}</div>
    </div>
  );
};

export default ManagerSystemSettingsTab; 