"use client";

import { useState, useEffect, type FormEvent, useRef } from 'react';
import { useRouter } from 'next/navigation';
import { signInWithEmailAndPassword, signInWithPopup, type UserCredential } from 'firebase/auth';
import { auth, googleProvider } from '../../lib/firebase';
import { useAuth } from '../../hooks/use-auth';
import { Button } from '../../components/ui/button';
import { Input } from '../../components/ui/input';
import { Label } from '../../components/ui/label';
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '../../components/ui/card';
import { useToast } from '../../hooks/use-toast';
import { Mail, Lock, LogIn } from 'lucide-react';
import { Separator } from '../../components/ui/separator';
import { SplashScreen } from '../../components/splash-screen';
import { requestNotificationPermission } from '../../lib/firebase-messaging';
import { useTranslation } from 'react-i18next';
import Head from 'next/head';

// Google Icon SVG
const GoogleIcon = () => (
  <svg className="mr-2 h-4 w-4" viewBox="0 0 24 24" fill="currentColor">
    <path
      d="M22.56 12.25c0-.78-.07-1.53-.2-2.25H12v4.26h5.92c-.26 1.37-1.04 2.53-2.21 3.31v2.77h3.57c2.08-1.92 3.28-4.74 3.28-8.09z"
      fill="#4285F4"
    />
    <path
      d="M12 23c2.97 0 5.46-.98 7.28-2.66l-3.57-2.77c-.98.66-2.23 1.06-3.71 1.06-2.86 0-5.29-1.93-6.16-4.53H2.18v2.84C3.99 20.53 7.7 23 12 23z"
      fill="#34A853"
    />
    <path
      d="M5.84 14.09c-.22-.66-.35-1.36-.35-2.09s.13-1.43.35-2.09V7.07H2.18C1.43 8.55 1 10.22 1 12s.43 3.45 1.18 4.93l2.85-2.22.81-.62z"
      fill="#FBBC05"
    />
    <path
      d="M12 5.38c1.62 0 3.06.56 4.21 1.64l3.15-3.15C17.45 2.09 14.97 1 12 1 7.7 1 3.99 3.47 2.18 7.07l3.66 2.84c.87-2.6 3.3-4.53 6.16-4.53z"
      fill="#EA4335"
    />
    <path d="M1 1h22v22H1z" fill="none" />
  </svg>
);

// SVG-флаги
const FLAGS: Record<string, string> = {
  he: '/img/il.svg',
  en: '/img/gb.svg', 
  es: '/img/es.svg',
  ru: '/img/ru.svg',
  ar: '/img/arab.svg',
};

const LANGUAGES = [
  { code: 'he', label: 'עברית' },
  { code: 'en', label: 'English' },
  { code: 'es', label: 'Español' },
  { code: 'ru', label: 'Русский' },
  { code: 'ar', label: 'العربية' },
];



export default function LoginPage() {
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [error, setError] = useState<string | null>(null);
  const [isLoading, setIsLoading] = useState(false);
  const [isGoogleLoading, setIsGoogleLoading] = useState(false);
  const [currentLanguage, setCurrentLanguage] = useState('he');
  const router = useRouter();
  const { user, loading: authLoading } = useAuth();
  const { toast } = useToast();
  const { t, i18n } = useTranslation();

  // Инициализация языка при загрузке
  useEffect(() => {
    const savedLanguage = localStorage.getItem('i18nextLng') || 'he';
    setCurrentLanguage(savedLanguage);
    if (i18n.language !== savedLanguage) {
      i18n.changeLanguage(savedLanguage);
    }
  }, [i18n]);

  // Обновление состояния при изменении языка
  useEffect(() => {
    setCurrentLanguage(i18n.language);
  }, [i18n.language]);

  const handleLanguageChange = (langCode: string) => {
    setCurrentLanguage(langCode);
    i18n.changeLanguage(langCode);
    localStorage.setItem('i18nextLng', langCode);
  };

  useEffect(() => {
    // console.log("[Login Page Effect] Auth loading:", authLoading, "User:", user);
    if (!authLoading && user) {
      // console.log("[Login Page Effect] User authenticated, redirecting to dashboard.");
      router.replace('/dashboard');
    }
  }, [user, authLoading, router]);

  const handleLogin = async (e: FormEvent) => {
    e.preventDefault();
    console.log("[Email Login][Step 1] Attempting email/password sign-in for email:", email);
    setIsLoading(true);
    setError(null);
    try {
      console.log("[Email Login][Step 2] Calling Firebase signInWithEmailAndPassword.");
      const userCredential = await signInWithEmailAndPassword(auth, email, password);
      console.log("[Email Login][Step 3] signInWithEmailAndPassword successful.");
      toast({
        title: "Login Successful",
        description: "Welcome back!",
      });

      // Request notification permission and get FCM token
      if (typeof window !== 'undefined') {
        try {
          const token = await requestNotificationPermission();
          if (token) {
            try {
              // Get device information
              const deviceInfo = {
                device_name: navigator.userAgent,
                client_device_type: /Mobile|Android|iPhone|iPad|iPod/i.test(navigator.userAgent) ? 'mobile' : 'desktop'
              };

              await fetch("/api/postgres/save-fcm-token", {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify({ 
                  email: userCredential.user.email, 
                  fcm_token: token,
                  ...deviceInfo
                }),
              });
            } catch (error) {
              console.error("Failed to save FCM token:", error);
            }
          }
        } catch (error) {
          console.error("Failed to request notification permission:", error);
        }
      }

      console.log("[Email Login][Step 4] Redirecting to dashboard.");
      router.push('/dashboard');
    } catch (err: any) {
      console.error("[Email Login][Step 3 - Error] Error during email/password sign-in:", err);
      let errorMessage = "Failed to login. Please check your credentials.";
      if (err.code === 'auth/user-not-found' || err.code === 'auth/wrong-password' || err.code === 'auth/invalid-credential') {
        errorMessage = "Invalid email or password.";
      } else if (err.code === 'auth/invalid-email') {
        errorMessage = "Please enter a valid email address.";
      } else if (err.code === 'auth/unauthorized-domain') {
        errorMessage = "This domain is not authorized for Firebase operations. Please check your Firebase project's authentication settings.";
        console.error("[Email Login] Firebase Error: Unauthorized domain. Ensure the current domain is added to the Firebase console (Authentication > Sign-in method > Authorized domains). Current domain might be: " + (typeof window !== "undefined" ? window.location.hostname : "unknown"));
      }
      setError(errorMessage);
      toast({
        variant: "destructive",
        title: "Login Failed",
        description: errorMessage,
      });
      console.log("[Email Login][Step 4 - Error] Displayed error message and toast.");
    } finally {
      console.log("[Email Login][Step 5] Email/password sign-in process finished.");
      setIsLoading(false);
    }
  };

  const handleGoogleSignIn = async () => {
    console.log("[Google Sign-In][Step 1] Attempting Google Sign-In.");
    setIsGoogleLoading(true);
    setError(null);
    try {
      console.log("[Google Sign-In][Step 2] Calling Firebase signInWithPopup.");
      const result: UserCredential = await signInWithPopup(auth, googleProvider);
      console.log("[Google Sign-In][Step 3] signInWithPopup successful. User:", result.user);
      toast({
        title: "Google Sign-In Successful",
        description: `Welcome, ${result.user.displayName || result.user.email}!`,
      });
      console.log("[Google Sign-In][Step 4] Redirecting to dashboard.");
      router.push('/dashboard');
    } catch (err: any) {
      console.error("[Google Sign-In][Step 3 - Error] Error during Google Sign-In:", err);
      let errorMessage = "Failed to sign in with Google. Please try again.";
      let toastVariant: "default" | "destructive" = "destructive";
      let toastTitle = "Google Sign-In Failed";

      if (err.code === 'auth/popup-closed-by-user') {
        errorMessage = "Google Sign-In was cancelled. You can try signing in again if you wish.";
        toastVariant = "default"; 
        toastTitle = "Google Sign-In Cancelled"; 
        console.log("[Google Sign-In][Info] Popup closed by user.");
      } else if (err.code === 'auth/account-exists-with-different-credential') {
        errorMessage = "An account already exists with the same email address but different sign-in credentials. Sign in using a provider associated with this email address.";
        console.log("[Google Sign-In][Error] Account exists with different credential.");
      } else if (err.code === 'auth/unauthorized-domain') {
        errorMessage = "This domain is not authorized for Firebase operations. Please check your Firebase project's authentication settings.";
        console.error("[Google Sign-In] Firebase Error: Unauthorized domain. Ensure the current domain is added to the Firebase console (Authentication > Sign-in method > Authorized domains). Current domain might be: " + (typeof window !== "undefined" ? window.location.hostname : "unknown"));
      }
       else {
         console.log("[Google Sign-In][Error] Unknown error:", err.code, err.message);
      }
      setError(errorMessage);
      toast({
        variant: toastVariant,
        title: toastTitle,
        description: errorMessage,
      });
      console.log("[Google Sign-In][Step 4 - Error] Displayed error message and toast.");
    } finally {
      console.log("[Google Sign-In][Step 5] Google Sign-In process finished.");
      setIsGoogleLoading(false);
    }
  };


  if (authLoading || (!authLoading && user)) {
    // console.log("[Login Page Render] Auth loading or user already authenticated, showing splash/loader.");
    return <SplashScreen />;
  }
  
  // console.log("[Login Page Render] Rendering login form.");
  return (
    <>
      <Head>
        <link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet" />
      </Head>
      <div className="main-container bg-gradient-to-br from-background to-secondary/10">
        <Card className="auth-card shadow-2xl w-full max-w-md">
          <img src="/img/top.svg" alt="Login Top" className="w-full object-contain mb-2" style={{maxWidth: "100%"}} />
          {/* Language Switcher (centered, Material style, no dropdown) */}
          <div className="flex justify-center items-center gap-4 p-2 mb-2">
            {LANGUAGES.map(lang => (
              <button
                key={lang.code}
                onClick={() => handleLanguageChange(lang.code)}
                className={`rounded-full border-2 border-transparent hover:border-blue-400 focus:outline-none flex items-center justify-center ${currentLanguage === lang.code ? 'border-blue-600 bg-blue-50' : ''}`}
                style={{ width: 40, height: 40, background: 'none' }}
                aria-label={lang.label}
                title={lang.label}
              >
                <img 
                  src={FLAGS[lang.code]} 
                  alt={lang.label}
                  className="w-7 h-7 object-contain"
                />
              </button>
            ))}
          </div>
          <CardHeader className="text-center">
            <p className="text-sm text-muted-foreground">{t('auth.loginTitle', 'Sign in to access your MS4SC dashboard.')}</p>
          </CardHeader>
          <CardContent className="space-y-6">
            <form onSubmit={handleLogin} className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="email">{t('common.email', 'Email')}</Label>
                <div className="relative">
                  <Mail className="absolute left-3 top-1/2 h-5 w-5 -translate-y-1/2 text-muted-foreground" />
                  <Input
                    id="email"
                    type="email"
                    placeholder={t('common.email', 'you@example.com')}
                    value={email}
                    onChange={(e) => setEmail(e.target.value)}
                    required
                    className="pl-10"
                    aria-label={t('common.email', 'Email Address')}
                    disabled={isLoading || isGoogleLoading}
                  />
                </div>
              </div>
              <div className="space-y-2">
                <Label htmlFor="password">{t('common.password', 'Password')}</Label>
                <div className="relative">
                  <Lock className="absolute left-3 top-1/2 h-5 w-5 -translate-y-1/2 text-muted-foreground" />
                  <Input
                    id="password"
                    type="password"
                    placeholder={t('common.password', '••••••••')}
                    value={password}
                    onChange={(e) => setPassword(e.target.value)}
                    required
                    className="pl-10"
                    aria-label={t('common.password', 'Password')}
                    disabled={isLoading || isGoogleLoading}
                    autoComplete="current-password"
                  />
                </div>
              </div>
              {error && <p className="text-sm text-destructive">{t('auth.invalidCredentials', error)}</p>}
              <Button type="submit" className="w-full bg-[#1899D3] hover:bg-[#1479A3] text-white" disabled={isLoading || isGoogleLoading}>
                {isLoading ? (
                  <svg className="mr-2 h-4 w-4 animate-spin" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                    <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                    <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                  </svg>
                ) : (
                  <LogIn className="mr-2 h-4 w-4" />
                )}
                {t('auth.login', 'Sign In with Email')}
              </Button>
            </form>

            <div className="relative my-4">
              <div className="absolute inset-0 flex items-center">
                <Separator />
              </div>
              <div className="relative flex justify-center text-xs uppercase">
                <span className="bg-background px-2 text-muted-foreground">
                  {t('auth.orContinueWith', 'Or continue with')}
                </span>
              </div>
            </div>

            <Button variant="outline" className="w-full" onClick={handleGoogleSignIn} disabled={isLoading || isGoogleLoading}>
              {isGoogleLoading ? (
                 <svg className="mr-2 h-4 w-4 animate-spin" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                  <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                  <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                </svg>
              ) : (
                <GoogleIcon />
              )}
              {t('auth.google', 'Sign in with Google')}
            </Button>

            

          </CardContent>
          <CardFooter className="flex-col items-center justify-center text-sm pt-6">
            <p className="text-muted-foreground">
              powered by <a href="https://ms4sc.com" target="_blank" rel="noopener noreferrer">MS4SC</a>
            </p>
          </CardFooter>
        </Card>
      </div>
    </>
  );
}

