import { NextRequest, NextResponse } from 'next/server';
import { postgresClient } from '../../../../lib/postgresql-client';

export async function GET(request: Request) {
  const { searchParams } = new URL(request.url);
  const email = searchParams.get('email');
  const userIdParam = searchParams.get('user_id');

  if (!email && !userIdParam) {
    return NextResponse.json({ error: 'Missing email or user_id parameter' }, { status: 400 });
  }

  try {
    let userIds = [];
    let roleId = null;
    if (userIdParam) {
      // Для админа: только один user_id
      userIds = [parseInt(userIdParam, 10)];
      const userRes = await postgresClient.query('SELECT role_id FROM users WHERE user_id = $1', [userIdParam]);
      if (userRes.rows.length > 0) roleId = userRes.rows[0].role_id;
    } else if (email) {
      // Для обычного пользователя: все user_id по email
      const userRes = await postgresClient.query('SELECT user_id, role_id FROM users WHERE email = $1', [email]);
      userIds = userRes.rows.map(row => row.user_id);
      if (userRes.rows.length > 0) roleId = userRes.rows[0].role_id;
    }
    if (userIds.length === 0) {
      return NextResponse.json({ devices: [], roleId });
    }
    // Получить все device_id для пользователя(ей) из user_meters
    const metersRes = await postgresClient.query('SELECT device_id FROM user_meters WHERE user_id = ANY($1)', [userIds]);
    const deviceIds = metersRes.rows.map(row => row.device_id);
    if (deviceIds.length === 0) {
      return NextResponse.json({ devices: [], roleId });
    }
    // Получить информацию об устройствах
    const deviceQuery = `SELECT device_name, dev_eui, device_type FROM devices WHERE device_name = ANY($1)`;
    const deviceResult = await postgresClient.query(deviceQuery, [deviceIds]);
    const devices = deviceResult.rows.map(row => ({
      device_id: row.device_name,
      dev_eui: row.dev_eui || row.device_name,
      device_type: row.device_type || 'Device'
    }));
    return NextResponse.json({ devices, roleId });
  } catch (error: any) {
    console.error('Failed to fetch user devices:', error);
    return NextResponse.json({ error: `Failed to retrieve user devices: ${error.message || error}` }, { status: 500 });
  }
}

export async function POST(request: Request) {
  try {
    const { email, device_id } = await request.json();
    if (!email || !device_id) {
      return NextResponse.json({ error: 'Missing email or device_id' }, { status: 400 });
    }
    // Найти user_id по email
    const userRes = await postgresClient.query('SELECT user_id FROM users WHERE email = $1', [email]);
    if (userRes.rows.length === 0) {
      return NextResponse.json({ error: 'User not found' }, { status: 404 });
    }
    const user_id = userRes.rows[0].user_id;
    // Добавить связь user-device
    await postgresClient.query('INSERT INTO user_meters (user_id, device_id) VALUES ($1, $2) ON CONFLICT DO NOTHING', [user_id, device_id]);
    return NextResponse.json({ success: true });
  } catch (error: any) {
    console.error('Failed to add device to user:', error);
    return NextResponse.json({ error: error.message || 'Failed to add device to user' }, { status: 500 });
  }
}

export async function DELETE(request: Request) {
  try {
    const { email, device_id } = await request.json();
    if (!email || !device_id) {
      return NextResponse.json({ error: 'Missing email or device_id' }, { status: 400 });
    }
    // Найти user_id по email
    const userRes = await postgresClient.query('SELECT user_id FROM users WHERE email = $1', [email]);
    if (userRes.rows.length === 0) {
      return NextResponse.json({ error: 'User not found' }, { status: 404 });
    }
    const user_id = userRes.rows[0].user_id;
    // Удалить связь user-device
    await postgresClient.query('DELETE FROM user_meters WHERE user_id = $1 AND device_id = $2', [user_id, device_id]);
    return NextResponse.json({ success: true });
  } catch (error: any) {
    console.error('Failed to remove device from user:', error);
    return NextResponse.json({ error: error.message || 'Failed to remove device from user' }, { status: 500 });
  }
} 