import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Input } from "../components/ui/input";
import { Button } from "../components/ui/button";
import { Alert, AlertDescription, AlertTitle } from "../components/ui/alert";
import { AlertCircle, CheckCircle2 } from "lucide-react";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "../components/ui/tabs";
import { Label } from "../components/ui/label";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
} from "../components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "../components/ui/select";
import { useAuth } from '../hooks/use-auth';
import { useTranslation } from 'react-i18next';

interface SystemConfig {
  env: {
    MQTT_BROKER_URL: string;
    MQTT_APP_ID: string;
    NEXT_PUBLIC_API_URL: string;
    NEXT_PUBLIC_APP_NAME: string;
    NEXT_PUBLIC_MQTT_BROKER_URL: string;
    NEXT_PUBLIC_MQTT_APP_ID: string;
  };
  ui: {
    menu: {
      dashboard: string;
      devices: string;
      settings: string;
      profile: string;
    };
    messages: {
      success: string;
      error: string;
      warning: string;
      info: string;
    };
    alerts: {
      deviceUpdate: string;
      maxValueUpdate: string;
      valveControl: string;
    };
    tabs: {
      system: string;
      devices: string;
      settings: string;
    };
    titles: {
      mainDashboard: string;
      deviceDetails: string;
      systemConfig: string;
    };
  };
}

const defaultConfig: SystemConfig = {
  env: {
    MQTT_BROKER_URL: '',
    MQTT_APP_ID: '',
    NEXT_PUBLIC_API_URL: '',
    NEXT_PUBLIC_APP_NAME: '',
    NEXT_PUBLIC_MQTT_BROKER_URL: '',
    NEXT_PUBLIC_MQTT_APP_ID: '',
  },
  ui: {
    menu: {
      dashboard: 'Dashboard',
      devices: 'Devices',
      settings: 'Settings',
      profile: 'Profile',
    },
    messages: {
      success: 'Operation completed successfully',
      error: 'An error occurred',
      warning: 'Warning',
      info: 'Information',
    },
    alerts: {
      deviceUpdate: 'Device updated successfully',
      maxValueUpdate: 'Max value updated successfully',
      valveControl: 'Valve control command sent',
    },
    tabs: {
      system: 'System',
      devices: 'Devices',
      settings: 'Settings',
    },
    titles: {
      mainDashboard: 'Main Dashboard',
      deviceDetails: 'Device Details',
      systemConfig: 'System Configuration',
    },
  },
};

interface User {
  user_id: number;
  user_name: string;
  email: string;
  phone: string;
  role_id: number;
  device_id: string | null;
  uid: string;
  payment_history_id: number | null;
}

interface UserModalProps {
  isOpen: boolean;
  onClose: () => void;
  user: User | null;
  onUpdate: (userId: number, updatedData: Partial<User>) => Promise<void>;
}

const UserModal: React.FC<UserModalProps> = ({ isOpen, onClose, user, onUpdate }) => {
  const { t, i18n } = useTranslation();
  const isRTL = i18n.dir() === 'rtl';
  const [formData, setFormData] = useState<Partial<User>>({});
  const [isUpdating, setIsUpdating] = useState(false);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    if (user) {
      setFormData({
        email: user.email,
        phone: user.phone,
        role_id: user.role_id,
        device_id: user.device_id,
        uid: user.uid,
        payment_history_id: user.payment_history_id,
      });
    }
  }, [user]);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!user) return;

    setIsUpdating(true);
    setError(null);

    try {
      await onUpdate(user.user_id, formData);
      onClose();
    } catch (err: any) {
      setError(err.message || t('systemConfigurationCard.failedToUpdateUser', 'Failed to update user'));
    } finally {
      setIsUpdating(false);
    }
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-[600px] max-h-[90vh] p-0" dir={isRTL ? 'rtl' : 'ltr'}>
        <DialogHeader className={`px-6 pt-6 ${isRTL ? 'text-right' : ''}`}>
          <DialogTitle>{t('systemConfigurationCard.editUserTitle', { userName: user?.user_name })}</DialogTitle>
          <DialogDescription>
            {t('systemConfigurationCard.editUserDescription', 'Modify user information')}
          </DialogDescription>
        </DialogHeader>
        <div className="px-6 pb-6">
          <Card className="border-0 shadow-lg">
            <CardContent className="p-0">
              <form onSubmit={handleSubmit} className="space-y-4">
                <div className="max-h-[60vh] overflow-y-auto pr-4 space-y-4">
                  <div className="space-y-2">
                    <Label htmlFor="user_id" className={isRTL ? 'text-right' : ''}>{t('systemConfigurationCard.userId', 'User ID')}</Label>
                    <Input id="user_id" value={user?.user_id || ""} disabled className="bg-muted" dir={isRTL ? 'rtl' : 'ltr'} />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="user_name" className={isRTL ? 'text-right' : ''}>{t('systemConfigurationCard.userName', 'Username')}</Label>
                    <Input id="user_name" value={user?.user_name || ""} disabled className="bg-muted" dir={isRTL ? 'rtl' : 'ltr'} />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="email" className={isRTL ? 'text-right' : ''}>{t('common.email', 'Email')}</Label>
                    <Input id="email" value={formData.email || ""} onChange={(e) => setFormData({ ...formData, email: e.target.value })} type="email" required dir={isRTL ? 'rtl' : 'ltr'} />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="phone" className={isRTL ? 'text-right' : ''}>{t('common.phone', 'Phone')}</Label>
                    <Input id="phone" value={formData.phone || ""} onChange={(e) => setFormData({ ...formData, phone: e.target.value })} type="tel" dir={isRTL ? 'rtl' : 'ltr'} />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="role" className={isRTL ? 'text-right' : ''}>{t('common.role', 'Role')}</Label>
                    <Select value={formData.role_id?.toString()} onValueChange={(value) => setFormData({ ...formData, role_id: parseInt(value) })}>
                      <SelectTrigger dir={isRTL ? 'rtl' : 'ltr'}>
                        <SelectValue placeholder={t('systemConfigurationCard.selectRole', 'Select role')} />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="0">{t('systemConfigurationCard.superuser', 'Superuser')}</SelectItem>
                        <SelectItem value="1">{t('systemConfigurationCard.admin', 'Admin')}</SelectItem>
                        <SelectItem value="2">{t('systemConfigurationCard.user', 'User')}</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="device_id" className={isRTL ? 'text-right' : ''}>{t('systemConfigurationCard.deviceId', 'Device ID')}</Label>
                    <Input id="device_id" value={formData.device_id || ""} onChange={(e) => setFormData({ ...formData, device_id: e.target.value })} dir={isRTL ? 'rtl' : 'ltr'} />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="uid" className={isRTL ? 'text-right' : ''}>{t('systemConfigurationCard.uid', 'UID')}</Label>
                    <Input id="uid" value={formData.uid || ""} onChange={(e) => setFormData({ ...formData, uid: e.target.value })} dir={isRTL ? 'rtl' : 'ltr'} />
                  </div>
                  <div className="space-y-2">
                    <Label htmlFor="payment_history_id" className={isRTL ? 'text-right' : ''}>{t('systemConfigurationCard.paymentHistoryId', 'Payment History ID')}</Label>
                    <Input id="payment_history_id" value={formData.payment_history_id || ""} onChange={(e) => setFormData({ ...formData, payment_history_id: parseInt(e.target.value) || null })} type="number" dir={isRTL ? 'rtl' : 'ltr'} />
                  </div>
                </div>
                {error && (
                  <Alert variant="destructive" className="mt-4">
                    <AlertCircle className="h-4 w-4" />
                    <AlertTitle>{t('common.error', 'Error')}</AlertTitle>
                    <AlertDescription>{error}</AlertDescription>
                  </Alert>
                )}
                <div className={`flex justify-end space-x-2 pt-4 border-t ${isRTL ? 'flex-row-reverse' : ''}`}>
                  <Button type="button" variant="outline" onClick={onClose}>
                    {t('common.cancel', 'Cancel')}
                  </Button>
                  <Button type="submit" disabled={isUpdating}>
                    {isUpdating ? t('systemConfigurationCard.updating', 'Updating...') : t('systemConfigurationCard.updateUser', 'Update User')}
                  </Button>
                </div>
              </form>
            </CardContent>
          </Card>
        </div>
      </DialogContent>
    </Dialog>
  );
};

function useIsMobile(breakpoint = 768) {
  const [isMobile, setIsMobile] = useState(false);
  useEffect(() => {
    const check = () => setIsMobile(window.matchMedia(`(max-width: ${breakpoint}px)`).matches);
    check();
    window.addEventListener('resize', check);
    return () => window.removeEventListener('resize', check);
  }, [breakpoint]);
  return isMobile;
}

const SystemConfigurationCard: React.FC = () => {
  const { t, i18n } = useTranslation();
  const isRTL = i18n.dir() === 'rtl';
  const isMobile = useIsMobile();
  const { userPermissions } = useAuth();
  const [config, setConfig] = useState<SystemConfig>(defaultConfig);
  const [activeTab, setActiveTab] = useState('env');
  const [alertMessage, setAlertMessage] = useState<{ type: 'success' | 'error'; message: string } | null>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [users, setUsers] = useState<User[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [selectedUser, setSelectedUser] = useState<User | null>(null);
  const [search, setSearch] = useState('');
  const [sortKey, setSortKey] = useState<'user_name'|'email'|'phone'|'role_id'>('user_name');
  const [sortDir, setSortDir] = useState<'asc'|'desc'>('asc');

  const fetchUsers = async () => {
    try {
      const response = await fetch('/api/postgres/users');
      if (!response.ok) throw new Error(t('systemConfigurationCard.failedToFetchUsers', 'Failed to fetch users'));
      const data = await response.json();
      setUsers(data.users);
    } catch (err: any) {
      setError(err.message || t('systemConfigurationCard.failedToFetchUsers', 'Failed to fetch users'));
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    const loadConfig = async () => {
      try {
        setIsLoading(true);
        const response = await fetch('/api/system-config');
        if (response.ok) {
          const data = await response.json();
          setConfig(data);
        } else {
          throw new Error(t('systemConfigurationCard.failedToLoadConfiguration', 'Failed to load configuration'));
        }
      } catch (error) {
        console.error('Failed to load system configuration:', error);
        setAlertMessage({
          type: 'error',
          message: t('systemConfigurationCard.failedToLoadConfiguration', 'Failed to load configuration'),
        });
      } finally {
        setIsLoading(false);
      }
    };

    loadConfig();
  }, [t]);

  useEffect(() => {
    fetchUsers();
  }, [t]);

  const handleEnvUpdate = async () => {
    try {
      const response = await fetch('/api/system-config', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          type: 'env',
          data: config.env,
        }),
      });

      if (response.ok) {
        setAlertMessage({
          type: 'success',
          message: t('systemConfigurationCard.environmentVariablesUpdated', 'Environment variables updated successfully'),
        });
      } else {
        throw new Error(t('systemConfigurationCard.failedToUpdateEnvironment', 'Failed to update environment variables'));
      }
    } catch (error) {
      setAlertMessage({
        type: 'error',
        message: t('systemConfigurationCard.failedToUpdateEnvironment', 'Failed to update environment variables'),
      });
    }
  };

  const handleUIUpdate = async () => {
    try {
      const response = await fetch('/api/system-config', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          type: 'ui',
          data: config.ui,
        }),
      });

      if (response.ok) {
        setAlertMessage({
          type: 'success',
          message: t('systemConfigurationCard.uiConfigurationUpdated', 'UI configuration updated successfully'),
        });
      } else {
        throw new Error(t('systemConfigurationCard.failedToUpdateUi', 'Failed to update UI configuration'));
      }
    } catch (error) {
      setAlertMessage({
        type: 'error',
        message: t('systemConfigurationCard.failedToUpdateUi', 'Failed to update UI configuration'),
      });
    }
  };

  const handleUpdateUser = async (userId: number, updatedData: Partial<User>) => {
    try {
      const response = await fetch('/api/postgres/users', {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ userId, ...updatedData }),
      });

      if (!response.ok) throw new Error(t('systemConfigurationCard.failedToUpdateUser', 'Failed to update user'));
      
      // Refresh users list
      await fetchUsers();
    } catch (err: any) {
      throw new Error(err.message || t('systemConfigurationCard.failedToUpdateUser', 'Failed to update user'));
    }
  };

  // Filter and sort users
  const filteredUsers = users
    .filter(u =>
      u.user_name.toLowerCase().includes(search.toLowerCase()) ||
      u.email.toLowerCase().includes(search.toLowerCase()) ||
      (u.phone || '').toLowerCase().includes(search.toLowerCase())
    )
    .sort((a, b) => {
      const aVal = a[sortKey] || '';
      const bVal = b[sortKey] || '';
      if (aVal < bVal) return sortDir === 'asc' ? -1 : 1;
      if (aVal > bVal) return sortDir === 'asc' ? 1 : -1;
      return 0;
    });

  // Для select
  const tabOptions = [
    { value: 'env', label: t('systemConfigurationCard.environmentVariables', 'Environment Variables') },
    { value: 'ui', label: t('systemConfigurationCard.uiConfiguration', 'UI Configuration') },
    { value: 'users', label: t('systemConfigurationCard.systemUsers', 'System Users') },
  ];

  if (isLoading) {
    return (
      <Card style={{ width: '90vw', maxWidth: 900 }}>
        <CardHeader>
          <CardTitle className={isRTL ? 'text-right' : ''}>{t('systemConfigurationCard.systemConfiguration', 'System Configuration')}</CardTitle>
        </CardHeader>
        <CardContent className="px-1 sm:px-4">
          <div className="space-y-4">
            <div className="h-8 w-full animate-pulse bg-muted rounded" />
            <div className="h-8 w-full animate-pulse bg-muted rounded" />
            <div className="h-8 w-full animate-pulse bg-muted rounded" />
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card style={{ width: '90vw', maxWidth: 900 }}>
      <CardHeader>
        <CardTitle className={isRTL ? 'text-right' : ''}>{t('systemConfigurationCard.systemConfiguration', 'System Configuration')}</CardTitle>
      </CardHeader>
      <CardContent className="px-1 sm:px-4">
        {alertMessage && (
          <Alert variant={alertMessage.type === 'success' ? 'default' : 'destructive'} className="mb-4">
            {alertMessage.type === 'success' ? (
              <CheckCircle2 className="h-4 w-4" />
            ) : (
              <AlertCircle className="h-4 w-4" />
            )}
            <AlertTitle>
              {alertMessage.type === 'success' ? t('common.success', 'Success') : t('common.error', 'Error')}
            </AlertTitle>
            <AlertDescription>{alertMessage.message}</AlertDescription>
          </Alert>
        )}

        <div className="overflow-x-auto w-full">
          {/* Удаляем форму поиска и выпадающий список tabOptions */}
          {/* Контент секций */}
          <Tabs value={activeTab} onValueChange={setActiveTab} dir={isRTL ? 'rtl' : 'ltr'}>
            <TabsContent value="env" className={activeTab === 'env' ? 'block space-y-4' : 'hidden'}>
              <div className="grid gap-4">
                {Object.entries(config.env)
                  .filter(([key]) => !['MQTT_BROKER_URL', 'MQTT_APP_ID', 'NEXT_PUBLIC_API_URL', 'NEXT_PUBLIC_APP_NAME'].includes(key))
                  .map(([key, value]) => (
                    <div key={key} className="space-y-2">
                      <label className={`text-sm font-medium ${isRTL ? 'text-right' : ''}`}>{key}</label>
                      <Input
                        value={value}
                        onChange={(e) =>
                          setConfig({
                            ...config,
                            env: { ...config.env, [key]: e.target.value },
                          })
                        }
                        disabled={!userPermissions?.can_change_env}
                        dir={isRTL ? 'rtl' : 'ltr'}
                        className="w-full box-border shadow-lg"
                      />
                    </div>
                  ))}
              </div>
              <Button onClick={handleEnvUpdate} className="w-full box-border shadow-lg" disabled={!userPermissions?.can_change_env} style={{ background: '#4194a2', color: '#fff' }}>
                {t('systemConfigurationCard.updateEnvironmentVariables', 'Update Environment Variables')}
              </Button>
            </TabsContent>
            <TabsContent value="ui" className={activeTab === 'ui' ? 'block space-y-4' : 'hidden'}>
              <div className="grid gap-4">
                {Object.entries(config.ui).map(([section, items]) => (
                  <div key={section} className="space-y-2">
                    <h3 className={`text-lg font-medium capitalize ${isRTL ? 'text-right' : ''}`}>{section}</h3>
                    <div className="grid gap-2">
                      {Object.entries(items).map(([key, value]) => (
                        <div key={key} className="space-y-1">
                          <label className={`text-sm font-medium ${isRTL ? 'text-right' : ''}`}>{key}</label>
                          <Input
                            value={value}
                            onChange={(e) =>
                              setConfig({
                                ...config,
                                ui: {
                                  ...config.ui,
                                  [section]: {
                                    ...config.ui[section as keyof typeof config.ui],
                                    [key]: e.target.value,
                                  },
                                },
                              })
                            }
                            disabled={!userPermissions?.can_change_ui}
                            dir={isRTL ? 'rtl' : 'ltr'}
                            className="w-full box-border shadow-lg"
                          />
                        </div>
                      ))}
                    </div>
                  </div>
                ))}
              </div>
              <Button onClick={handleUIUpdate} className="w-full box-border shadow-lg" disabled={!userPermissions?.can_change_ui} style={{ background: '#4194a2', color: '#fff' }}>
                {t('systemConfigurationCard.updateUiConfiguration', 'Update UI Configuration')}
              </Button>
            </TabsContent>
            <TabsContent value="users" className={activeTab === 'users' ? 'block space-y-4' : 'hidden'}>
              {/* Удалено: Search and Sort Controls */}
              <div className="relative overflow-x-auto w-full">
                <table className="w-full min-w-[600px] text-sm text-left">
                  <thead className="text-xs uppercase bg-muted">
                    <tr>
                      <th className={`px-3 sm:px-6 py-3 whitespace-nowrap ${isRTL ? 'text-right' : ''}`}>{t('systemConfigurationCard.username', 'Username')}</th>
                      <th className={`px-3 sm:px-6 py-3 whitespace-nowrap ${isRTL ? 'text-right' : ''}`}>{t('common.email', 'Email')}</th>
                      <th className={`px-3 sm:px-6 py-3 whitespace-nowrap ${isRTL ? 'text-right' : ''}`}>{t('common.phone', 'Phone')}</th>
                      <th className={`px-3 sm:px-6 py-3 whitespace-nowrap ${isRTL ? 'text-right' : ''}`}>{t('common.role', 'Role')}</th>
                      <th className={`px-3 sm:px-6 py-3 whitespace-nowrap ${isRTL ? 'text-right' : ''}`}>{t('systemConfigurationCard.deviceId', 'Device ID')}</th>
                    </tr>
                  </thead>
                  <tbody>
                    {filteredUsers.map((user) => (
                      <tr
                        key={user.user_id}
                        className="border-b hover:bg-muted/50 cursor-pointer"
                        onClick={() => setSelectedUser(user)}
                      >
                        <td className={`px-3 sm:px-6 py-4 font-medium ${isRTL ? 'text-right' : ''}`}>{user.user_name}</td>
                        <td className={`px-3 sm:px-6 py-4 ${isRTL ? 'text-right' : ''}`}>{user.email}</td>
                        <td className={`px-3 sm:px-6 py-4 ${isRTL ? 'text-right' : ''}`}>{user.phone}</td>
                        <td className={`px-3 sm:px-6 py-4 ${isRTL ? 'text-right' : ''}`}>
                          {user.role_id === 0 ? t('systemConfigurationCard.superuser', 'Superuser') : user.role_id === 1 ? t('systemConfigurationCard.admin', 'Admin') : t('systemConfigurationCard.user', 'User')}
                        </td>
                        <td className={`px-3 sm:px-6 py-4 ${isRTL ? 'text-right' : ''}`}>{user.device_id || 'N/A'}</td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
              {selectedUser && (
                <UserModal
                  isOpen={!!selectedUser}
                  onClose={() => setSelectedUser(null)}
                  user={selectedUser}
                  onUpdate={handleUpdateUser}
                />
              )}
            </TabsContent>
          </Tabs>
        </div>
      </CardContent>
    </Card>
  );
};

export default SystemConfigurationCard;