import React, { useEffect, useState, useMemo } from 'react';
import { Card, CardHeader, CardTitle, CardContent } from '../components/ui/card';
import { Button } from '../components/ui/button';
import { useTranslation } from 'react-i18next';
import DeviceDetailModal from './DeviceDetailModal';
import DeviceAdminEditModal from './DeviceAdminEditModal';
import { useAuth } from '../hooks/use-auth';

function useIsMobile(breakpoint = 768) {
  const [isMobile, setIsMobile] = useState(false);
  useEffect(() => {
    const check = () => setIsMobile(window.matchMedia(`(max-width: ${breakpoint}px)`).matches);
    check();
    window.addEventListener('resize', check);
    return () => window.removeEventListener('resize', check);
  }, [breakpoint]);
  return isMobile;
}

const DeviceManagementTable: React.FC = () => {
  const { t, i18n } = useTranslation();
  const isRTL = i18n.dir() === 'rtl';
  const isMobile = useIsMobile();
  const { userRoleId } = useAuth();
  const [devices, setDevices] = useState<any[]>([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState<string | null>(null);
  const [search, setSearch] = useState('');
  const [selectedMobileDevice, setSelectedMobileDevice] = useState<string>('');
  const [selectedDevice, setSelectedDevice] = useState<any | null>(null);
  const [modalOpen, setModalOpen] = useState(false);

  // Расширенный маппинг типа устройства к ключу перевода
  const DEVICE_TYPE_MAP: Record<string, string> = {
    'waterMeter': 'devices.waterMeter',
    'electricMeter': 'devices.electricMeter',
    'Water meter': 'devices.waterMeter',
    'Electric meter': 'devices.electricMeter',
    'Water Meter': 'devices.waterMeter',
    'Electric Meter': 'devices.electricMeter',
  };

  const getDeviceTypeLabel = (type: string) => {
    if (!type) return '';
    const key = DEVICE_TYPE_MAP[type.trim()] || type.trim();
    return t(key, type);
  };

  // Выбор endpoint в зависимости от роли
  const getEndpoint = () => userRoleId === 0 ? '/api/postgres/all-user-devices-admin' : '/api/postgres/all-user-devices';

  useEffect(() => {
    fetch(getEndpoint())
      .then(res => {
        if (!res.ok) {
          return res.json().then(err => {
            throw new Error(err.details || err.error || t('systemAdminTab.failedToFetchDevices', 'Failed to fetch devices'));
          });
        }
        return res.json();
      })
      .then(data => setDevices(data.devices || []))
      .catch(err => {
        setError(err.message || t('systemAdminTab.failedToFetchDevices', 'Failed to fetch devices'));
        setDevices([]);
      });
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [t, userRoleId]);

  // Фильтрация устройств по поиску
  const filteredMobileDevices = useMemo(() => {
    if (!search.trim()) return devices;
    const s = search.trim().toLowerCase();
    return devices.filter(device =>
      (device.device_name && device.device_name.toLowerCase().includes(s)) ||
      (device.device_type && device.device_type.toLowerCase().includes(s)) ||
      (device.user_id && String(device.user_id).toLowerCase().includes(s))
    );
  }, [devices, search]);

  const handleRemoveDevice = async (deviceId: number) => {
    if (!window.confirm(t('systemAdminTab.confirmRemoveDevice', 'Are you sure you want to remove this device?'))) return;
    try {
      setLoading(true);
      const res = await fetch(`/api/postgres/delete-user-device`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ id: deviceId })
      });
      if (!res.ok) throw new Error(t('systemAdminTab.failedToRemoveDevice', 'Failed to remove device'));
      setSuccess(t('systemAdminTab.deviceRemoved', 'Device removed successfully'));
      // Refetch devices after removal
      fetch(getEndpoint())
        .then(res => res.json())
        .then(data => setDevices(data.devices || []))
        .catch(() => setDevices([]));
    } catch (err: any) {
      setError(err.message || t('systemAdminTab.failedToRemoveDevice', 'Failed to remove device'));
    } finally {
      setLoading(false);
    }
  };

  return (
    <Card className="mt-6" style={isMobile ? { width: '90vw', maxWidth: '90vw' } : {}}>
      <CardHeader>
        <CardTitle className={isRTL ? 'text-right' : ''}>{t('systemAdminTab.deviceManagement', 'Device Management')}</CardTitle>
      </CardHeader>
      <CardContent className="px-1 sm:px-4">
        {error && <div className="text-red-500 mb-2">{error}</div>}
        {success && <div className="text-green-600 mb-2">{success}</div>}
        
        {/* Мобильная форма поиска устройства */}
        {isMobile ? (
          <div className="w-full flex flex-col gap-2 mb-4">
            <input
              type="text"
              className="w-full border rounded px-3 py-2 text-sm"
              placeholder={t('dashboard.searchPlaceholder', 'Search by device number')}
              value={search}
              onChange={e => setSearch(e.target.value)}
            />
            <select
              className="w-full border rounded px-3 py-2 text-sm"
              value={selectedMobileDevice}
              onChange={e => {
                setSelectedMobileDevice(e.target.value);
                const dev = filteredMobileDevices.find(d => d.device_name === e.target.value);
                if (dev) {
                  setSelectedDevice(dev);
                  setModalOpen(true);
                }
              }}
            >
              <option value="">{t('dashboard.selectDevicePlaceholder', 'Select a device...')}</option>
              {filteredMobileDevices.map(device => (
                <option key={device.device_name} value={device.device_name}>
                  {device.device_name}
                </option>
              ))}
            </select>
            {/* Модальное окно устройства */}
            {modalOpen && selectedDevice && (
              userRoleId === 0 ? (
                <DeviceAdminEditModal
                  isOpen={modalOpen}
                  onClose={() => setModalOpen(false)}
                  device={selectedDevice}
                  onSave={async (updated) => {
                    await fetch('/api/postgres/update-device', {
                      method: 'POST',
                      headers: { 'Content-Type': 'application/json' },
                      body: JSON.stringify(updated)
                    });
                    setModalOpen(false);
                    fetch(getEndpoint())
                      .then(res => res.json())
                      .then(data => setDevices(data.devices || []));
                  }}
                  onDelete={async (dev) => {
                    await handleRemoveDevice(dev.id);
                    setModalOpen(false);
                  }}
                />
              ) : (
                <DeviceDetailModal
                  isOpen={modalOpen}
                  onClose={() => setModalOpen(false)}
                  device={{
                    deviceId: selectedDevice.device_name,
                    maxValue: selectedDevice.max_value ?? null,
                    twentyFourHourValveState: selectedDevice.valve_state ?? null,
                    twentyFourHourCurrentReading: selectedDevice.current_read ?? null,
                    device_type: selectedDevice.device_type ?? null,
                    roleId: typeof selectedDevice.roleId === 'number' ? selectedDevice.roleId : 0,
                    isAdminDevice: true,
                    dev_eui: selectedDevice.dev_eui || '',
                  }}
                  onMaxValueUpdate={() => {
                    fetch(getEndpoint())
                      .then(res => res.json())
                      .then(data => setDevices(data.devices || []));
                  }}
                />
              )
            )}
          </div>
        ) : (
          // Десктопная таблица
          <div className="overflow-x-auto w-full">
            <table className="w-full min-w-[600px] text-sm border-collapse">
              <thead>
                <tr>
                  {/* <th className="p-2 border whitespace-nowrap">{t('systemAdminTab.id', 'ID')}</th> */}
                  <th className="p-2 border whitespace-nowrap">{t('systemAdminTab.userId', 'User ID')}</th>
                  <th className="p-2 border whitespace-nowrap">{t('systemAdminTab.deviceId', 'Device ID')}</th>
                  <th className="p-2 border whitespace-nowrap">{t('systemAdminTab.uid', 'UID')}</th>
                  <th className="p-2 border whitespace-nowrap">{t('systemAdminTab.fcmToken', 'FCM Token')}</th>
                  <th className="p-2 border whitespace-nowrap">{t('systemAdminTab.deviceName', 'Device Name')}</th>
                  <th className="p-2 border whitespace-nowrap">{t('systemAdminTab.deviceType', 'Device Type')}</th>
                  <th className="p-2 border whitespace-nowrap">{t('systemAdminTab.actions', 'Actions')}</th>
                </tr>
              </thead>
              <tbody>
                {devices.map(device => (
                  <tr key={`${device.device_name}-${device.user_id || ''}`} className="hover:bg-muted/50 cursor-pointer" onClick={() => { setSelectedDevice(device); setModalOpen(true); }}>
                    {/* <td className="p-2 border text-center">{device.id}</td> */}
                    <td className="p-2 border text-center">{device.user_id}</td>
                    <td className="p-2 border text-center">{device.device_id}</td>
                    <td className="p-2 border text-center">{device.uid}</td>
                    <td className="p-2 border max-w-[120px] truncate text-center">{device.fcm_token || '-'}</td>
                    <td className="p-2 border text-center font-bold underline">{device.device_name}</td>
                    <td className="p-2 border text-center">{getDeviceTypeLabel(device.device_type)}</td>
                    <td className="p-2 border text-center" onClick={e => e.stopPropagation()}>
                      <Button variant="destructive" size="sm" onClick={() => handleRemoveDevice(device.id)} disabled={loading} className="w-full box-border">
                        {t('systemAdminTab.remove', 'Remove')}
                      </Button>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
            {/* Модальное окно устройства для десктопа */}
            {modalOpen && selectedDevice && (
              userRoleId === 0 ? (
                <DeviceAdminEditModal
                  isOpen={modalOpen}
                  onClose={() => setModalOpen(false)}
                  device={selectedDevice}
                  onSave={async (updated) => {
                    await fetch('/api/postgres/update-device', {
                      method: 'POST',
                      headers: { 'Content-Type': 'application/json' },
                      body: JSON.stringify(updated)
                    });
                    setModalOpen(false);
                    fetch(getEndpoint())
                      .then(res => res.json())
                      .then(data => setDevices(data.devices || []));
                  }}
                  onDelete={async (dev) => {
                    await handleRemoveDevice(dev.id);
                    setModalOpen(false);
                  }}
                />
              ) : (
                <DeviceDetailModal
                  isOpen={modalOpen}
                  onClose={() => setModalOpen(false)}
                  device={{
                    deviceId: selectedDevice.device_name,
                    maxValue: selectedDevice.max_value ?? null,
                    twentyFourHourValveState: selectedDevice.valve_state ?? null,
                    twentyFourHourCurrentReading: selectedDevice.current_read ?? null,
                    device_type: selectedDevice.device_type ?? null,
                    roleId: typeof selectedDevice.roleId === 'number' ? selectedDevice.roleId : 0,
                    isAdminDevice: true,
                    dev_eui: selectedDevice.dev_eui || '',
                  }}
                  onMaxValueUpdate={() => {
                    fetch(getEndpoint())
                      .then(res => res.json())
                      .then(data => setDevices(data.devices || []));
                  }}
                />
              )
            )}
          </div>
        )}
      </CardContent>
    </Card>
  );
};

export default DeviceManagementTable;