import React, { useState } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from '../components/ui/dialog';
import { Button } from '../components/ui/button';
import { Input } from '../components/ui/input';
import { useTranslation } from 'react-i18next';

interface DeviceAdminEditModalProps {
  isOpen: boolean;
  onClose: () => void;
  device: any;
  onSave: (updated: any) => void;
  onDelete: (device: any) => void;
}

const DeviceAdminEditModal: React.FC<DeviceAdminEditModalProps> = ({ isOpen, onClose, device, onSave, onDelete }) => {
  const { t } = useTranslation();
  const typeMap: Record<string, string> = {
    'modem': 'modem',
    'Modem': 'modem',
    'water_meter': 'water_meter',
    'Water Meter': 'water_meter',
    'waterMeter': 'water_meter',
    'Счётчик воды': 'water_meter',
    'מונה מים': 'water_meter',
    'electric_meter': 'electric_meter',
    'Electric Meter': 'electric_meter',
    'electricMeter': 'electric_meter',
    'Счётчик электроэнергии': 'electric_meter',
    'מונה חשמל': 'electric_meter',
    'other': 'other',
    'Other': 'other',
    'Другое': 'other',
    'אחר': 'other',
  };

  const getMappedType = (raw: string) => typeMap[raw] || '';

  const [form, setForm] = useState({
    user_id: device.user_id || '',
    device_id: device.device_id || '',
    uid: device.uid || '',
    fcm_token: device.fcm_token || '',
    device_name: device.device_name || '',
    device_type: getMappedType(device.device_type || ''),
  });
  const [loading, setLoading] = useState(false);
  const isMobile = typeof window !== 'undefined' && window.innerWidth <= 768;

  const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setForm({ ...form, [e.target.name]: e.target.value });
  };

  const handleSave = async () => {
    setLoading(true);
    await onSave(form);
    setLoading(false);
  };

  const handleDelete = async () => {
    setLoading(true);
    await onDelete(device);
    setLoading(false);
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent style={{ width: isMobile ? '95vw' : undefined, maxWidth: 600, height: isMobile ? '80vh' : undefined, maxHeight: 600, overflowY: 'auto' }}>
        <DialogHeader>
          <DialogTitle>{t('deviceAdminEditModal.title', 'Edit Device')}</DialogTitle>
        </DialogHeader>
        <div className="grid gap-4 py-2">
          <label className="block">
            {t('systemAdminTab.userId', 'User ID')}
            <Input name="user_id" value={form.user_id} onChange={handleChange} className="mt-1 shadow-lg" />
          </label>
          <label className="block">
            {t('systemAdminTab.deviceId', 'Device ID')}
            <Input name="device_id" value={form.device_id} onChange={handleChange} className="mt-1 shadow-lg" />
          </label>
          <label className="block">
            {t('systemAdminTab.uid', 'UID')}
            <Input name="uid" value={form.uid} onChange={handleChange} className="mt-1 shadow-lg" />
          </label>
          <label className="block">
            {t('systemAdminTab.fcmToken', 'FCM Token')}
            <Input name="fcm_token" value={form.fcm_token} onChange={handleChange} className="mt-1 shadow-lg" />
          </label>
          <label className="block">
            {t('systemAdminTab.deviceName', 'Device Name')}
            <Input name="device_name" value={form.device_name} onChange={handleChange} className="mt-1 shadow-lg" />
          </label>
          <label className="block">
            {t('systemAdminTab.deviceType', 'Device Type')}
            <select
              name="device_type"
              value={form.device_type}
              onChange={e => setForm({ ...form, device_type: e.target.value })}
              className="mt-1 w-full border rounded px-2 py-2 shadow-lg"
            >
              <option value="">{t('deviceAdminEditModal.selectDeviceType', 'Select device type')}</option>
              <option value="modem">{t('deviceTypes.modem', 'Modem')}</option>
              <option value="water_meter">{t('deviceTypes.water_meter', 'Water meter')}</option>
              <option value="electric_meter">{t('deviceTypes.electric_meter', 'Electric meter')}</option>
              <option value="other">{t('deviceTypes.other', 'Other')}</option>
            </select>
          </label>
        </div>
        <DialogFooter className="flex flex-row gap-2 justify-between mt-4">
          <Button variant="destructive" onClick={handleDelete} disabled={loading} className="shadow-lg">
            {t('common.delete', 'Delete')}
          </Button>
          <Button onClick={handleSave} disabled={loading} className="shadow-lg">
            {t('common.save', 'Save')}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
};

export default DeviceAdminEditModal; 