import React, { useState, useEffect } from 'react';
import { useTranslation } from 'react-i18next';
import { Button } from '../components/ui/button';
import { Input } from '../components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '../components/ui/select';
import { Card, CardHeader, CardTitle, CardContent } from '../components/ui/card';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '../components/ui/dialog';
import { useAuth } from '../hooks/use-auth';

const LANGUAGES = [
  { value: 'en', label: 'English' },
  { value: 'he', label: 'עברית' },
  { value: 'es', label: 'Español' },
  { value: 'ru', label: 'Русский' },
  { value: 'ar', label: 'العربية' },
];

const UserSettingsTab: React.FC = () => {
  const { t, i18n } = useTranslation();
  const isRTL = i18n.dir() === 'rtl';
  const { user } = useAuth();
  const [language, setLanguage] = useState('he');
  const [userData, setUserData] = useState<any>(null);
  const [family, setFamily] = useState<any[]>([]);
  const [showAddUser, setShowAddUser] = useState(false);
  const [phone, setPhone] = useState('');
  const [phoneChanged, setPhoneChanged] = useState(false);
  const [saving, setSaving] = useState(false);
  
  // Form state for adding user (updated to match manager-panel)
  const [newUserForm, setNewUserForm] = useState({
    f_name: '',
    s_name: '',
    p_id: '',
    phone: '',
    email: '',
    role_id: '2',
    relationship: ''
  });
  const [addingUser, setAddingUser] = useState(false);
  const [addUserError, setAddUserError] = useState<string | null>(null);
  
  // State for device selection
  const [userDevices, setUserDevices] = useState<any[]>([]);
  const [selectedDeviceIdsForNewUser, setSelectedDeviceIdsForNewUser] = useState<string[]>([]);
  
  // State for relationship types
  const [relationshipTypes, setRelationshipTypes] = useState<any[]>([]);

  // Состояние для модального окна редактирования членов семьи
  const [editFamilyModalOpen, setEditFamilyModalOpen] = useState(false);
  const [editingFamilyMember, setEditingFamilyMember] = useState<any>(null);
  const [editingFamilyData, setEditingFamilyData] = useState<any>({});
  const [savingFamily, setSavingFamily] = useState(false);

  // Fetch user data and family members
  useEffect(() => {
    if (user?.email) {
      fetch(`/api/postgres/users?email=${encodeURIComponent(user.email)}`)
        .then(res => res.json())
        .then(data => {
          setUserData(data);
          setLanguage(data.language || 'he');
          setPhone(data.phone || '');
          // Загружаем членов семьи только после получения userData
          if (data.p_id) {
            fetch(`/api/postgres/family-members?p_id=${encodeURIComponent(data.p_id)}`)
              .then(res => res.json())
              .then(data => setFamily(data.familyMembers || []));
          }
        });
    }
  }, [user]);

  // Load user devices
  useEffect(() => {
    if (userData?.user_id) {
      fetch(`/api/postgres/user-device?user_id=${userData.user_id}`)
        .then(res => res.json())
        .then(data => setUserDevices(data.devices || []));
    }
  }, [userData?.user_id]);

  // Load relationship types
  useEffect(() => {
    fetch('/api/postgres/relationship-types')
      .then(res => res.json())
      .then(data => setRelationshipTypes(data.relationshipTypes || []));
  }, []);

  // Reset device selection when form closes
  useEffect(() => {
    if (!showAddUser) {
      setSelectedDeviceIdsForNewUser([]);
    }
  }, [showAddUser]);

  // Handle language change
  const handleLanguageChange = async (lang: string) => {
    setLanguage(lang);
    i18n.changeLanguage(lang);
    // Save to DB
    if (user?.email) {
      await fetch('/api/postgres/users', {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email: user.email, language: lang })
      });
    }
  };

  // Сохранение телефона
  const handleSavePhone = async () => {
    if (!user?.email) return;
    setSaving(true);
    await fetch('/api/postgres/users', {
      method: 'PUT',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ email: user.email, phone })
    });
    setPhoneChanged(false);
    setSaving(false);
  };

  // Updated function for adding new user (from manager-panel)
  const handleAddUser = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!newUserForm.f_name || !newUserForm.s_name || !newUserForm.email || !newUserForm.p_id) {
      setAddUserError('Все поля обязательны для заполнения');
      return;
    }

    // Use selected devices or fallback to first device
    let deviceIdsToUse = selectedDeviceIdsForNewUser && selectedDeviceIdsForNewUser.length > 0
      ? selectedDeviceIdsForNewUser
      : userDevices.length > 0
        ? [userDevices[0].device_id]
        : [];

    if (deviceIdsToUse.length === 0) {
      setAddUserError('У вас нет устройств. Сначала добавьте устройство.');
      return;
    }
    
    setAddingUser(true);
    setAddUserError(null);
    
    try {
      const response = await fetch('/api/postgres/users', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          ...newUserForm,
          ...(deviceIdsToUse.length === 1
            ? { device_id: deviceIdsToUse[0] }
            : { device_ids: deviceIdsToUse })
        })
      });
      
      if (response.ok) {
        // Reset form
        setNewUserForm({
          f_name: '',
          s_name: '',
          p_id: '',
          phone: '',
          email: '',
          role_id: '2',
          relationship: ''
        });
        setShowAddUser(false);
        setSelectedDeviceIdsForNewUser([]);
        // Refresh family members
        if (userData.p_id) {
          const res = await fetch(`/api/postgres/family-members?p_id=${encodeURIComponent(userData.p_id)}`);
          const data = await res.json();
          setFamily(data.familyMembers || []);
        }
      } else {
        const errorData = await response.json();
        setAddUserError(errorData.error || 'Ошибка создания пользователя');
      }
    } catch (error) {
      setAddUserError('Ошибка создания пользователя');
    }
    setAddingUser(false);
  };

  // Функции для работы с модальным окном редактирования членов семьи
  const openEditFamilyModal = (member: any) => {
    //console.log('DEBUG openEditFamilyModal', member);
    setEditingFamilyMember(member);
    // Нормализуем c_sex из Hebrew в 'male'/'female'
    let normalizedSex = member.c_sex;
    if (member.c_sex === 'בן') normalizedSex = 'male';
    if (member.c_sex === 'בת') normalizedSex = 'female';
    
    setEditingFamilyData({
      c_name: member.c_name || '',
      c_age: member.c_age || '',
      c_sex: normalizedSex || 'male'
    });
    setEditFamilyModalOpen(true);
  };

  const handleSaveFamilyMember = async () => {
    if (!editingFamilyMember || !userData?.p_id) return;
    setSavingFamily(true);
    await fetch('/api/postgres/family-members', {
      method: 'PUT',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        c_id: editingFamilyMember.c_id,
        p_id: userData.p_id,
        ...editingFamilyData,
      }),
    });
    setSavingFamily(false);
    setEditFamilyModalOpen(false);
    // Обновить список семьи
    fetch(`/api/postgres/family-members?p_id=${encodeURIComponent(userData.p_id)}`)
      .then(res => res.json())
      .then(data => setFamily(data.familyMembers || []));
  };

  // Проверка языка ввода для полей (кроме email)
  const isHebrew = language === 'he';
  const showHebrewInputWarning = !isHebrew;

  return (
    <Card className="w-full">
      <CardHeader>
        <CardTitle className="text-lg sm:text-xl">{t('settings.title')}</CardTitle>
      </CardHeader>
      <CardContent className="space-y-4 sm:space-y-6">
        {/* Language selection */}
        <div>
          <label className="block mb-2 font-medium text-sm sm:text-base">{t('settings.language')}</label>
          <Select value={language} onValueChange={handleLanguageChange}>
            <SelectTrigger className="w-full sm:w-48">
              <SelectValue />
            </SelectTrigger>
            <SelectContent>
              {LANGUAGES.map(lang => (
                <SelectItem key={lang.value} value={lang.value}>{lang.label}</SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>
        {/* User data */}
        {userData && (
          <div className="space-y-2 text-sm sm:text-base">
            <div><b>{t('settings.f_name')}:</b> {userData.f_name}</div>
            <div><b>{t('settings.s_name')}:</b> {userData.s_name}</div>
            <div><b>{t('settings.p_id')}:</b> {userData.p_id}</div>
            <div><b>{t('settings.email')}:</b> {userData.email}</div>
            <div>
              <b>{t('settings.phone')}:</b>
              <Input
                className="inline-block w-full sm:w-48 ml-0 sm:ml-2 mt-1 sm:mt-0"
                value={phone}
                onChange={e => {
                  setPhone(e.target.value);
                  setPhoneChanged(e.target.value !== (userData?.phone || ''));
                }}
                disabled={!(userData.role_id === 1 || userData.role_id === 2)}
                lang={isHebrew ? 'he' : undefined}
                dir={isHebrew ? 'rtl' : 'ltr'}
              />
              {phoneChanged && (
                <Button size="sm" className="ml-0 sm:ml-2 mt-2 sm:mt-0 bg-[#1899D3] hover:bg-[#1479A3] text-white" onClick={handleSavePhone} disabled={saving}>
                  {saving ? t('common.loading', 'Saving...') : t('common.save', 'Save')}
                </Button>
              )}
            </div>
            {showHebrewInputWarning && (
              <div className="text-xs sm:text-sm text-red-500">{t('settings.input_hebrew_warning', 'Please change your input language to Hebrew for all fields except email.')}</div>
            )}
          </div>
        )}
        {/* Family members */}
        <div>
          <div className="font-medium mb-2 text-sm sm:text-base">{t('settings.family')}</div>
          <div className="border rounded p-2 sm:p-3">
            {family.length === 0 ? (
              <div className="text-muted-foreground text-xs sm:text-sm">{t('settings.no_family')}</div>
            ) : (
              <ul className="space-y-2">
                {family.map(member => (
                  <li key={member.id} className="flex flex-col sm:flex-row sm:items-center sm:justify-between space-y-1 sm:space-y-0">
                    <span className="text-sm sm:text-base">{member.f_name} {member.s_name} ({member.relation})</span>
                    <Button size="sm" variant="outline" className="w-full sm:w-auto border-[#1899D3] text-[#1899D3] hover:bg-[#1899D3] hover:text-white" onClick={() => openEditFamilyModal(member)}>{t('settings.edit')}</Button>
                  </li>
                ))}
              </ul>
            )}
            <Button size="sm" className="mt-2 w-full sm:w-auto bg-[#1899D3] hover:bg-[#1479A3] text-white">{t('settings.add_family')}</Button>
          </div>
        </div>
        {/* Add user to device */}
        <div>
          <div className="font-medium mb-2 text-sm sm:text-base">{t('settings.add_user')}</div>
          <Button onClick={() => setShowAddUser(true)} className="w-full sm:w-auto bg-[#1899D3] hover:bg-[#1479A3] text-white">{t('settings.add_user_btn')}</Button>
          {showAddUser && (
            <form onSubmit={handleAddUser} className="mt-2 space-y-2 border rounded p-2 sm:p-3">
              {/* Device selection for authenticated user */}
              {userDevices.length > 0 && (
                <div className="pt-1">
                  <div className="font-medium text-xs sm:text-sm mb-1">{t('devices.device', 'Device')}</div>
                  <div className="grid grid-cols-1 sm:grid-cols-2 gap-2">
                    {userDevices.map((device) => {
                      const devId = device.device_id;
                      const checked = selectedDeviceIdsForNewUser.includes(devId);
                      return (
                        <label key={`user_dev_${devId}`} className="flex items-center gap-2 border rounded p-2">
                          <input
                            type="checkbox"
                            checked={checked}
                            onChange={(e) => {
                              setSelectedDeviceIdsForNewUser(prev => {
                                if (e.target.checked) return Array.from(new Set([...prev, devId]));
                                return prev.filter(id => id !== devId);
                              });
                            }}
                          />
                          <span>{devId}</span>
                        </label>
                      );
                    })}
                  </div>
                  <div className="text-xs text-muted-foreground mt-1">{t('devices.device', 'Device')}: {t('common.select', 'Select')} 1+</div>
                </div>
              )}
              <Input 
                placeholder={t('settings.f_name')} 
                value={newUserForm.f_name}
                onChange={(e) => setNewUserForm({...newUserForm, f_name: e.target.value})}
                lang={isHebrew ? 'he' : undefined} 
                dir={isHebrew ? 'rtl' : 'ltr'} 
                required
              />
              <Input 
                placeholder={t('settings.s_name')} 
                value={newUserForm.s_name}
                onChange={(e) => setNewUserForm({...newUserForm, s_name: e.target.value})}
                lang={isHebrew ? 'he' : undefined} 
                dir={isHebrew ? 'rtl' : 'ltr'} 
                required
              />
              <Input 
                placeholder={t('settings.p_id')} 
                value={newUserForm.p_id}
                onChange={(e) => setNewUserForm({...newUserForm, p_id: e.target.value})}
                lang={isHebrew ? 'he' : undefined} 
                dir={isHebrew ? 'rtl' : 'ltr'} 
                required
              />
              <Input 
                placeholder={t('settings.phone')} 
                value={newUserForm.phone}
                onChange={(e) => setNewUserForm({...newUserForm, phone: e.target.value})}
                lang={isHebrew ? 'he' : undefined} 
                dir={isHebrew ? 'rtl' : 'ltr'} 
              />
              <Input 
                placeholder={t('settings.email')} 
                type="email" 
                value={newUserForm.email}
                onChange={(e) => setNewUserForm({...newUserForm, email: e.target.value})}
                required
              />
              <Select value={newUserForm.role_id} onValueChange={value => setNewUserForm({...newUserForm, role_id: value})}>
                <SelectTrigger className="w-full sm:w-48">
                  <SelectValue placeholder={t('settings.role')} />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="2">{t('settings.role2')}</SelectItem>
                  <SelectItem value="3">{t('settings.role3')}</SelectItem>
                </SelectContent>
              </Select>
              <Select value={newUserForm.relationship} onValueChange={(value) => setNewUserForm({...newUserForm, relationship: value})}>
                <SelectTrigger className="w-full sm:w-48">
                  <SelectValue placeholder={t('settings.relationship')} />
                </SelectTrigger>
                <SelectContent>
                  {relationshipTypes.map(relType => (
                    <SelectItem key={relType.relationship_name} value={relType.relationship_name}>
                      {String(t(`relationship.${relType.relationship_name}`, relType.relationship_name))}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <div className="flex gap-2">
                <Button 
                  type="submit" 
                  className="w-full sm:w-auto bg-[#1899D3] hover:bg-[#1479A3] text-white"
                  disabled={addingUser}
                >
                  {addingUser ? t('common.loading', 'Adding...') : t('settings.save')}
                </Button>
                <Button 
                  type="button" 
                  variant="outline"
                  onClick={() => setShowAddUser(false)}
                  disabled={addingUser}
                >
                  {t('common.cancel', 'Cancel')}
                </Button>
              </div>
              {addUserError && (
                <div className="text-red-500 text-sm">{addUserError}</div>
              )}
              {showHebrewInputWarning && (
                <div className="text-xs sm:text-sm text-red-500">{t('settings.input_hebrew_warning', 'Please change your input language to Hebrew for all fields except email.')}</div>
              )}
            </form>
          )}
        </div>
      </CardContent>
      
      {/* Модальное окно редактирования члена семьи */}
      <Dialog open={editFamilyModalOpen} onOpenChange={setEditFamilyModalOpen}>
        <DialogContent className="sm:max-w-[425px]">
          <DialogHeader>
            <DialogTitle>{t('settings.editFamilyMember', 'Редактировать члена семьи')}</DialogTitle>
          </DialogHeader>
          <div className="grid gap-4 py-4">
            <div className="grid grid-cols-4 items-center gap-4">
              <label htmlFor="c_name" className="text-right">
                {t('settings.name', 'Имя')}
              </label>
              <Input
                id="c_name"
                value={editingFamilyData.c_name || ''}
                onChange={(e) => setEditingFamilyData({...editingFamilyData, c_name: e.target.value})}
                className={`col-span-3 ${editingFamilyData.c_name ? 'ring-2 ring-[#0B674E] border-[#0B674E]' : ''}`}
                lang={isHebrew ? 'he' : undefined}
                dir={isHebrew ? 'rtl' : 'ltr'}
              />
            </div>
            <div className="grid grid-cols-4 items-center gap-4">
              <label htmlFor="c_age" className="text-right">
                {t('settings.age', 'Возраст')}
              </label>
              <Input
                id="c_age"
                type="number"
                value={editingFamilyData.c_age || ''}
                onChange={(e) => setEditingFamilyData({...editingFamilyData, c_age: e.target.value})}
                className={`col-span-3 ${editingFamilyData.c_age ? 'ring-2 ring-[#0B674E] border-[#0B674E]' : ''}`}
              />
            </div>
            <div className="grid grid-cols-4 items-center gap-4">
              <label htmlFor="c_sex" className="text-right">
                {t('settings.sex', 'Пол')}
              </label>
              <Select 
                value={editingFamilyData.c_sex || 'male'} 
                onValueChange={(value) => setEditingFamilyData({...editingFamilyData, c_sex: value})}
              >
                <SelectTrigger className={`col-span-3 ${editingFamilyData.c_sex ? 'ring-2 ring-[#0B674E] border-[#0B674E]' : ''}`} dir={isRTL ? 'rtl' : 'ltr'}>
                  <SelectValue placeholder={t('settings.selectSex', 'Выберите пол')} />
                </SelectTrigger>
                <SelectContent dir={isRTL ? 'rtl' : 'ltr'}>
                  <SelectItem value="male" className={isRTL ? 'text-right' : 'text-left'}>
                    {t('settings.male', 'Мужской')}
                  </SelectItem>
                  <SelectItem value="female" className={isRTL ? 'text-right' : 'text-left'}>
                    {t('settings.female', 'Женский')}
                  </SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
          <div className="flex justify-end space-x-2">
            <Button variant="outline" onClick={() => setEditFamilyModalOpen(false)}>
              {t('common.cancel', 'Отмена')}
            </Button>
            <Button onClick={handleSaveFamilyMember} disabled={savingFamily}>
              {savingFamily ? t('common.loading', 'Сохранение...') : t('common.save', 'Сохранить')}
            </Button>
          </div>
        </DialogContent>
      </Dialog>
    </Card>
  );
};

export default UserSettingsTab; 