declare var mqtt: any;

const MQTT_BROKER_URL = process.env.NEXT_PUBLIC_MQTT_BROKER_URL;
const MQTT_APP_ID = process.env.NEXT_PUBLIC_MQTT_APP_ID;
const FPORT = 2;

export const MQTT_COMMANDS = {
  "80%": "JiUAAFRg/w==",
  "85%": "JiUAADhAww==",
  "90%": "JiUAABwghw==",
  "100%": ["JiUAAA4QaQ==", "Jh8BRg=="]
};

let mqttClientInstance: any = null;

// --- MQTT Connection Status Management ---
let mqttConnected = false;
let statusListeners: ((connected: boolean) => void)[] = [];

function notifyStatusListeners(status: boolean) {
  mqttConnected = status;
  statusListeners.forEach((cb) => cb(status));
}

export function subscribeMqttStatus(cb: (connected: boolean) => void) {
  statusListeners.push(cb);
  // Immediately call with current status
  cb(mqttConnected);
  return () => {
    statusListeners = statusListeners.filter((fn) => fn !== cb);
  };
}

export function getMqttStatus() {
  return mqttConnected;
}

export const loadMqttLibrary = (): Promise<void> => {
  return new Promise((resolve, reject) => {
    if (typeof mqtt !== "undefined" && mqtt.connect) {
      resolve();
    } else {
      const script = document.createElement("script");
      script.src = "https://unpkg.com/mqtt/dist/mqtt.min.js";
      script.async = true;
      script.onload = () => {
        if (typeof mqtt !== "undefined" && mqtt.connect) {
          resolve();
        } else {
          reject(new Error("MQTT library loaded but mqtt.connect is not available."));
        }
      };
      script.onerror = (err) => {
        console.error("Failed to load MQTT script:", err);
        reject(err);
      };
      document.head.appendChild(script);
    }
  });
};

function setupMqttEventHandlers(client: any) {
  if (!client) return;
  client.on("connect", () => notifyStatusListeners(true));
  client.on("close", () => notifyStatusListeners(false));
  client.on("offline", () => notifyStatusListeners(false));
  client.on("error", () => notifyStatusListeners(false));
}

export const sendMqttCommand = (devEUI: string, commandData: string | string[]): Promise<void> => {
  return new Promise((resolve, reject) => {
    if (!mqttClientInstance) {
      reject(new Error("MQTT not initialized"));
      return;
    }

    const topic = `application/${MQTT_APP_ID}/device/${devEUI}/command/down`;
    const commands = Array.isArray(commandData) ? commandData : [commandData];
    
    let completed = 0;
    let hasError = false;

    commands.forEach(data => {
      const message = {
        devEui: devEUI,
        confirmed: true,
        fPort: FPORT,
        data: data
      };
      
      mqttClientInstance.publish(topic, JSON.stringify(message), (err?: Error) => {
        if (err) {
          console.error("MQTT command failed:", err);
          hasError = true;
        } else {
          //console.log("MQTT command sent:", message);
        }
        
        completed++;
        if (completed === commands.length) {
          if (hasError) {
            reject(new Error("One or more MQTT commands failed"));
          } else {
            resolve();
          }
        }
      });
    });
  });
};

export const connectAndSend = async (devEUI: string, commandData: string | string[]): Promise<void> => {
  if (!MQTT_BROKER_URL || !MQTT_APP_ID) {
    throw new Error("MQTT configuration missing");
  }

  if (!mqttClientInstance?.connected) {
    mqttClientInstance = mqtt.connect(MQTT_BROKER_URL);
    setupMqttEventHandlers(mqttClientInstance);
    await new Promise<void>((resolve, reject) => {
      mqttClientInstance.once("connect", () => resolve());
      mqttClientInstance.once("error", (err: Error) => reject(err));
    });
  }

  return sendMqttCommand(devEUI, commandData);
};

export const disconnectMqtt = () => {
  if (mqttClientInstance?.connected) {
    mqttClientInstance.end();
    mqttClientInstance = null;
    notifyStatusListeners(false);
  }
};

export function ensureMqttConnected() {
  if (!mqttClientInstance && MQTT_BROKER_URL) {
    mqttClientInstance = mqtt.connect(MQTT_BROKER_URL);
    setupMqttEventHandlers(mqttClientInstance);
  }
} 