import React, { useState, useEffect } from 'react';
import { useTranslation } from 'react-i18next';
import { Button } from '../components/ui/button';
import { Input } from '../components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '../components/ui/select';
import { Card, CardHeader, CardTitle, CardContent } from '../components/ui/card';
import { DropdownMenu, DropdownMenuTrigger, DropdownMenuContent, DropdownMenuItem } from '../components/ui/dropdown-menu';
import { useAuth } from '../hooks/use-auth';
import { Dialog, DialogTrigger, DialogContent, DialogHeader, DialogTitle, DialogFooter } from '../components/ui/dialog';

const LANGUAGES = [
  { value: 'en', label: 'English' },
  { value: 'he', label: 'עברית' },
  { value: 'es', label: 'Español' },
  { value: 'ru', label: 'Русский' },
  { value: 'ar', label: 'العربية' },
];

const UserSettingsTab: React.FC = () => {
  const { t, i18n } = useTranslation();
  const isRTL = i18n.dir() === 'rtl';
  const { userRoleId } = useAuth();
  const [users, setUsers] = useState<any[]>([]);
  const [selectedUser, setSelectedUser] = useState<string>('');
  const [userData, setUserData] = useState<any>(null);
  const [family, setFamily] = useState<any[]>([]);
  const [showAddUser, setShowAddUser] = useState(false);
  const [phone, setPhone] = useState('');
  const [phoneChanged, setPhoneChanged] = useState(false);
  const [saving, setSaving] = useState(false);
  const [language, setLanguage] = useState('he');
  const [searchUser, setSearchUser] = useState('');
  const [userDevices, setUserDevices] = useState<any[]>([]);
  const [allDevices, setAllDevices] = useState<any[]>([]);
  const [deviceActionLoading, setDeviceActionLoading] = useState(false);
  const [selectedDeviceToAdd, setSelectedDeviceToAdd] = useState<string>('');
  const [allUserIds, setAllUserIds] = useState<number[]>([]);
  const [adminSelectedUserId, setAdminSelectedUserId] = useState<number | null>(null);
  const [adminMode, setAdminMode] = useState(false);
  const [editUserData, setEditUserData] = useState<any>(null);
  const [userDataChanged, setUserDataChanged] = useState(false);
  const [userDevicesByUserId, setUserDevicesByUserId] = useState<Record<number, any[]>>({});
  const [addDeviceError, setAddDeviceError] = useState<string | null>(null);
  const [relatedUserInfo, setRelatedUserInfo] = useState<any | null>(null);
  // Для множественного выбора устройств при добавлении пользователя
  const [selectedDeviceIdsForNewUser, setSelectedDeviceIdsForNewUser] = useState<string[]>([]);
  const selectableDevicesForNewUser = React.useMemo(() => {
    const ids = new Set<string>();
    if (selectedUser && allUserIds.length > 0) {
      for (const uid of allUserIds) {
        (userDevicesByUserId[uid] || []).forEach(d => { if (d?.device_id) ids.add(d.device_id); });
      }
    }
    if (userData?.user_id && userDevicesByUserId[userData.user_id]) {
      (userDevicesByUserId[userData.user_id] || []).forEach(d => { if (d?.device_id) ids.add(d.device_id); });
    }
    return Array.from(ids);
  }, [selectedUser, allUserIds, userDevicesByUserId, userData?.user_id]);

  // Полные объекты устройств из userDevicesByUserId по всем user_id выбранного пользователя
  const selectableDeviceObjects = React.useMemo(() => {
    const seen = new Set<string>();
    const result: { device_id: string; device_type?: string | null }[] = [];
    allUserIds.forEach(uid => {
      const list = userDevicesByUserId[uid] || [];
      list.forEach(d => {
        if (d?.device_id && !seen.has(d.device_id)) {
          seen.add(d.device_id);
          result.push({ device_id: d.device_id, device_type: d.device_type });
        }
      });
    });
    return result;
  }, [allUserIds, userDevicesByUserId]);

  // Сброс выбора устройств при закрытии формы
  useEffect(() => {
    if (!showAddUser) {
      setSelectedDeviceIdsForNewUser([]);
    }
  }, [showAddUser]);

  // Состояние для способов оплаты
  const [paymentMethods, setPaymentMethods] = useState<string[]>([]);
  // Состояние для формы добавления пользователя
  const [newUserForm, setNewUserForm] = useState({
    f_name: '',
    s_name: '',
    p_id: '',
    phone: '',
    email: '',
    role_id: '2',
    relationship: ''
  });
  const [addingUser, setAddingUser] = useState(false);
  const [addUserError, setAddUserError] = useState<string | null>(null);
  // удалено дублирующее объявление showAddUser
  const [relationshipTypes, setRelationshipTypes] = useState<any[]>([]);

  const PAYMENT_METHODS = React.useMemo(() => [
    { value: 'credit_card', label: t('paymentMethods.credit_card', 'Кредитная карта') },
    { value: 'bank_transfer', label: t('paymentMethods.bank_transfer', 'Банковский перевод') },
    { value: 'cash', label: t('paymentMethods.cash', 'Наличные') },
  ], [t, i18n.language]);

  const isHebrew = language === 'he';
  const showHebrewInputWarning = !isHebrew;
  const isAdmin = userRoleId === 0;

  const [editFamilyModalOpen, setEditFamilyModalOpen] = useState(false);
  const [editingFamilyMember, setEditingFamilyMember] = useState<any>(null);
  const [editingFamilyData, setEditingFamilyData] = useState<any>(null);
  const [savingFamily, setSavingFamily] = useState(false);

  useEffect(() => {
    fetch('/api/postgres/users')
      .then(res => res.json())
      .then(data => setUsers(data.users || []));
    fetch('/api/postgres/all-devices')
      .then(res => res.json())
      .then(data => setAllDevices(data.devices || []));
    // Типы связей
    fetch('/api/postgres/relationship-types')
      .then(res => res.json())
      .then(data => setRelationshipTypes(data.relationshipTypes || []));
  }, []);

  useEffect(() => {
    if (selectedUser) {
      fetch(`/api/postgres/users?email=${encodeURIComponent(selectedUser)}`)
        .then(res => res.json())
        .then(data => {
          setUserData(data);
          setLanguage(data.language || 'he');
          setPhone(data.phone || '');
          if (data.p_id) {
            fetch(`/api/postgres/family-members?p_id=${encodeURIComponent(data.p_id)}`)
              .then(res => res.json())
              .then(payload => {
                setFamily(payload.familyMembers || []);
                setRelatedUserInfo(payload.relatedUser || null);
              });
          } else {
            setFamily([]);
          }
        });
      // Получить все user_id для email
      fetch(`/api/postgres/users?all_user_ids_for_email=${encodeURIComponent(selectedUser)}`)
        .then(res => res.json())
        .then(data => {
          const ids = (data.users || []).map((u: any) => u.user_id);
          setAllUserIds(ids);
        });
    } else {
      setUserData(null);
      setFamily([]);
      setUserDevices([]);
      setAllUserIds([]);
    }
  }, [selectedUser]);

  // Получение устройств для всех user_id выбранного пользователя (без зависимости от adminMode)
  useEffect(() => {
    if (allUserIds.length > 0) {
      const fetchAll = async () => {
        const result: Record<number, any[]> = {};
        for (const uid of allUserIds) {
          try {
            const res = await fetch(`/api/postgres/user-device?user_id=${uid}`);
            const data = await res.json();
            result[uid] = data.devices || [];
          } catch {
            result[uid] = [];
          }
        }
        setUserDevicesByUserId(result);
      };
      fetchAll();
    } else {
      setUserDevicesByUserId({});
    }
  }, [allUserIds, selectedUser]);

  useEffect(() => {
    if (userData && isAdmin) {
      setEditUserData({
        f_name: userData.f_name || '',
        s_name: userData.s_name || '',
        email: userData.email || '',
        p_id: userData.p_id || '',
        phone: userData.phone || '',
      });
      setUserDataChanged(false);
    } else {
      setEditUserData(null);
      setUserDataChanged(false);
    }
  }, [userData, isAdmin]);

  // Загружать payment_methods при выборе пользователя
  useEffect(() => {
    if (userData && userData.payment_methods) {
      try {
        const arr = JSON.parse(userData.payment_methods);
        setPaymentMethods(Array.isArray(arr) ? arr : []);
      } catch {
        setPaymentMethods([]);
      }
    } else {
      setPaymentMethods([]);
    }
  }, [userData]);

  const handleLanguageChange = async (lang: string) => {
    setLanguage(lang);
    if (selectedUser) {
      await fetch('/api/postgres/users', {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email: selectedUser, language: lang })
      });
    }
  };

  const handleSavePhone = async () => {
    if (!selectedUser) return;
    setSaving(true);
    await fetch('/api/postgres/users', {
      method: 'PUT',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ email: selectedUser, phone })
    });
    setPhoneChanged(false);
    setSaving(false);
  };

  const handleEditUserChange = (field: string, value: string) => {
    setEditUserData((prev: any) => ({ ...prev, [field]: value }));
    setUserDataChanged(true);
  };

  const handleSaveUserData = async () => {
    if (!selectedUser || !editUserData) return;
    setSaving(true);
    await fetch('/api/postgres/users', {
      method: 'PUT',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        email: selectedUser,
        ...editUserData,
      }),
    });
    setUserDataChanged(false);
    setSaving(false);
    // Обновить userData после сохранения
    fetch(`/api/postgres/users?email=${encodeURIComponent(selectedUser)}`)
      .then(res => res.json())
      .then(data => setUserData(data));
  };

  // Сохранить payment_methods
  const handleSavePaymentMethods = async () => {
    if (!selectedUser) return;
    setSaving(true);
    const body: any = { payment_methods: paymentMethods };
    if (userData?.user_id) {
      body.userId = userData.user_id;
    } else {
      body.email = selectedUser;
    }
    await fetch('/api/postgres/users', {
      method: 'PUT',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(body),
    });
    setSaving(false);
    // Обновить userData после сохранения
    fetch(`/api/postgres/users?email=${encodeURIComponent(selectedUser)}`)
      .then(res => res.json())
      .then(data => setUserData(data));
  };

  const DEVICE_TYPE_MAP: Record<string, string> = {
    'waterMeter': 'devices.waterMeter',
    'electricMeter': 'devices.electricMeter',
    'Water meter': 'devices.waterMeter',
    'Electric meter': 'devices.electricMeter',
    'Water Meter': 'devices.waterMeter',
    'Electric Meter': 'devices.electricMeter',
  };
  const getDeviceTypeLabel = (type: string) => {
    if (!type) return '';
    const key = DEVICE_TYPE_MAP[type.trim()] || type.trim();
    return t(key, type);
  };

  // Для добавления устройства используем user_id и device_id
  const getCurrentUserId = () => allUserIds[0];

  const handleAddDevice = async () => {
    setAddDeviceError(null);
    const email = selectedUser;
    const device_id = selectedDeviceToAdd;
    if (!(adminMode && email && device_id)) return;
    setDeviceActionLoading(true);
    try {
      const res = await fetch('/api/postgres/user-device', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email, device_id }),
      });
      if (!res.ok) {
        const err = await res.json();
        setAddDeviceError(err?.error || 'Ошибка добавления устройства');
      } else {
        // обновить устройства
        const updated = { ...userDevicesByUserId };
        for (const uid of allUserIds) {
          const resp = await fetch(`/api/postgres/user-device?user_id=${uid}`);
          const data = await resp.json();
          updated[uid] = data.devices || [];
        }
        setUserDevicesByUserId(updated);
        setSelectedDeviceToAdd('');
      }
    } catch (e) {
      setAddDeviceError('Ошибка добавления устройства');
    }
    setDeviceActionLoading(false);
  };

  // Для удаления устройства используем user_id и device_id
  const handleRemoveDevice = async (device_id: string) => {
    setDeviceActionLoading(true);
    const email = selectedUser;
    try {
      await fetch('/api/postgres/user-device', {
        method: 'DELETE',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ email, device_id }),
      });
      // обновить устройства
      const updated = { ...userDevicesByUserId };
      for (const uid of allUserIds) {
        const resp = await fetch(`/api/postgres/user-device?user_id=${uid}`);
        const data = await resp.json();
        updated[uid] = data.devices || [];
      }
      setUserDevicesByUserId(updated);
    } catch {}
    setDeviceActionLoading(false);
  };

  useEffect(() => {
    setAdminMode(isAdmin);
  }, [isAdmin]);

  useEffect(() => {
    if (adminMode && allUserIds.length > 0 && !adminSelectedUserId) {
      setAdminSelectedUserId(allUserIds[0]);
    }
  }, [adminMode, allUserIds, adminSelectedUserId]);

  // Для userRoleId === 1: устройства выбранного пользователя
  useEffect(() => {
    if (userRoleId === 1 && userData?.user_id) {
      fetch(`/api/postgres/user-device?user_id=${userData.user_id}`)
        .then(res => res.json())
        .then(data => setUserDevices(data.devices || []));
    }
  }, [userRoleId, userData?.user_id]);

  // Функция для открытия модального окна редактирования члена семьи
  const openEditFamilyModal = (member: any) => {
    //console.log('DEBUG openEditFamilyModal', member);
    // Преобразование пола к 'male'/'female'
    let normalizedSex = member.c_sex;
    if (normalizedSex === 'בן') normalizedSex = 'male';
    if (normalizedSex === 'בת') normalizedSex = 'female';
    setEditingFamilyMember(member);
    setEditingFamilyData({
      c_name: member.c_name,
      c_sex: normalizedSex,
      c_age: member.c_age,
    });
    setEditFamilyModalOpen(true);
  };
  // Функция для сохранения изменений
  const handleSaveFamilyMember = async () => {
    if (!editingFamilyMember || !userData?.p_id) return;
    setSavingFamily(true);
    await fetch('/api/postgres/family-members', {
      method: 'PUT',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        c_id: editingFamilyMember.c_id,
        p_id: userData.p_id,
        ...editingFamilyData,
      }),
    });
    setSavingFamily(false);
    setEditFamilyModalOpen(false);
    // Обновить список семьи
    fetch(`/api/postgres/family-members?p_id=${encodeURIComponent(userData.p_id)}`)
      .then(res => res.json())
      .then(payload => {
        setFamily(payload.familyMembers || []);
        setRelatedUserInfo(payload.relatedUser || null);
      });
  };

  // Добавление нового пользователя к устройству выбранного пользователя
  const handleAddUser = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!newUserForm.f_name || !newUserForm.s_name || !newUserForm.email || !newUserForm.p_id) {
      setAddUserError('Все поля обязательны для заполнения');
      return;
    }

    // Собираем выбранные устройства
    let deviceIdsToUse = selectedDeviceIdsForNewUser && selectedDeviceIdsForNewUser.length > 0
      ? selectedDeviceIdsForNewUser
      : selectableDevicesForNewUser.length > 0
        ? [selectableDevicesForNewUser[0]]
        : [];

    if (deviceIdsToUse.length === 0) {
      setAddUserError('У выбранного пользователя нет устройств. Сначала добавьте устройство.');
      return;
    }

    setAddingUser(true);
    setAddUserError(null);

    try {
      const response = await fetch('/api/postgres/users', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          ...newUserForm,
          ...(deviceIdsToUse.length === 1
            ? { device_id: deviceIdsToUse[0] }
            : { device_ids: deviceIdsToUse })
        })
      });

      if (response.ok) {
        // Сброс формы
        setNewUserForm({
          f_name: '',
          s_name: '',
          p_id: '',
          phone: '',
          email: '',
          role_id: '2',
          relationship: ''
        });
        setShowAddUser(false);
        setSelectedDeviceIdsForNewUser([]);
        // Обновить список пользователей
        fetch('/api/postgres/users')
          .then(res => res.json())
          .then(data => setUsers(data.users || []));
      } else {
        const errorData = await response.json();
        setAddUserError(errorData.error || 'Ошибка создания пользователя');
      }
    } catch (error) {
      setAddUserError('Ошибка создания пользователя');
    }
    setAddingUser(false);
  };

  return (
    <Card className="w-full mb-40 shadow-lg">
      <CardHeader>
        <CardTitle className="text-lg sm:text-xl">{t('settings.title')}</CardTitle>
      </CardHeader>
      <CardContent className="space-y-4 sm:space-y-6">
        <div className="mb-4">
          <DropdownMenu>
            <DropdownMenuTrigger asChild>
              <button
                type="button"
                className={`w-full border rounded p-2 mb-2 bg-white ${isRTL ? 'text-right' : 'text-left'}`}
                style={{ direction: isRTL ? 'rtl' : 'ltr', textAlign: isRTL ? 'right' : 'left' }}
              >
                {selectedUser
                  ? (users.find(u => u.email === selectedUser)?.f_name || '') +
                    (users.find(u => u.email === selectedUser)?.s_name ? ' ' + users.find(u => u.email === selectedUser)?.s_name : '')
                  : t('systemAdminTab.allUsers', 'All Users')}
              </button>
            </DropdownMenuTrigger>
            <DropdownMenuContent className="max-h-60 overflow-y-auto shadow-lg">
              <input
                type="text"
                placeholder={t('systemAdminTab.searchUser', 'Search user...')}
                className="w-full border rounded p-2 mb-2"
                value={searchUser || ''}
                onChange={e => setSearchUser(e.target.value)}
                dir={isRTL ? 'rtl' : 'ltr'}
                style={{ textAlign: isRTL ? 'right' : 'left' }}
              />
              <DropdownMenuItem
                onSelect={() => setSelectedUser('')}
                className={selectedUser === '' ? `bg-accent text-accent-foreground ${isRTL ? 'text-right' : 'text-left'}` : isRTL ? 'text-right' : 'text-left'}
                style={{ direction: isRTL ? 'rtl' : 'ltr', textAlign: isRTL ? 'right' : 'left' }}
              >
                {t('systemAdminTab.allUsers', 'All Users')}
              </DropdownMenuItem>
              {Array.from(new Map(users.map(u => [u.email, u])).values())
                .filter(u => {
                  const search = (searchUser || '').toLowerCase();
                  return (
                    (u.f_name && u.f_name.toLowerCase().includes(search)) ||
                    (u.s_name && u.s_name.toLowerCase().includes(search))
                  );
                })
                .map((u, idx) => (
                  <DropdownMenuItem
                    key={u.email + '_' + (u.payment_history_id || idx)}
                    onSelect={() => setSelectedUser(u.email)}
                    className={selectedUser === u.email ? `bg-accent text-accent-foreground ${isRTL ? 'text-right' : 'text-left'}` : isRTL ? 'text-right' : 'text-left'}
                    style={{ direction: isRTL ? 'rtl' : 'ltr', textAlign: isRTL ? 'right' : 'left' }}
                  >
                    {(u.f_name || '') + (u.s_name ? ' ' + u.s_name : '') || u.email}
                  </DropdownMenuItem>
                ))}
            </DropdownMenuContent>
          </DropdownMenu>
        </div>
        {/* Language selection */}
        <div>
          <label className="block mb-2 font-medium text-sm sm:text-base">{t('settings.language')}</label>
          <Select value={language} onValueChange={handleLanguageChange}>
            <SelectTrigger className="w-full sm:w-48 shadow-lg">
              <SelectValue />
            </SelectTrigger>
            <SelectContent className="shadow-lg">
              {LANGUAGES.map(lang => (
                <SelectItem key={lang.value} value={lang.value}>{lang.label}</SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>
        {/* User data */}
        {userData && (
          <div className="space-y-2 text-sm sm:text-base">
            {isAdmin && editUserData ? (
              <>
                <div>
                  <b>{t('settings.f_name')}:</b>
                  <Input
                    className="inline-block w-full sm:w-48 ml-0 sm:ml-2 mt-1 sm:mt-0 shadow-lg"
                    value={editUserData.f_name}
                    onChange={e => handleEditUserChange('f_name', e.target.value)}
                    lang={isHebrew ? 'he' : undefined}
                    dir={isHebrew ? 'rtl' : 'ltr'}
                  />
                </div>
                <div>
                  <b>{t('settings.s_name')}:</b>
                  <Input
                    className="inline-block w-full sm:w-48 ml-0 sm:ml-2 mt-1 sm:mt-0 shadow-lg"
                    value={editUserData.s_name}
                    onChange={e => handleEditUserChange('s_name', e.target.value)}
                    lang={isHebrew ? 'he' : undefined}
                    dir={isHebrew ? 'rtl' : 'ltr'}
                  />
                </div>
                <div>
                  <b>{t('settings.p_id')}:</b>
                  <Input
                    className="inline-block w-full sm:w-48 ml-0 sm:ml-2 mt-1 sm:mt-0 shadow-lg"
                    value={editUserData.p_id}
                    onChange={e => handleEditUserChange('p_id', e.target.value)}
                    lang={isHebrew ? 'he' : undefined}
                    dir={isHebrew ? 'rtl' : 'ltr'}
                  />
                </div>
                <div>
                  <b>{t('settings.email')}:</b>
                  <Input
                    className="inline-block w-full sm:w-48 ml-0 sm:ml-2 mt-1 sm:mt-0 shadow-lg"
                    value={editUserData.email}
                    onChange={e => handleEditUserChange('email', e.target.value)}
                    lang={isHebrew ? 'he' : undefined}
                    dir={isHebrew ? 'rtl' : 'ltr'}
                  />
                </div>
                <div>
                  <b>{t('settings.phone')}:</b>
                  <Input
                    className="inline-block w-full sm:w-48 ml-0 sm:ml-2 mt-1 sm:mt-0 shadow-lg"
                    value={editUserData.phone}
                    onChange={e => handleEditUserChange('phone', e.target.value)}
                    lang={isHebrew ? 'he' : undefined}
                    dir={isHebrew ? 'rtl' : 'ltr'}
                  />
                </div>
                {userDataChanged && (
                  <Button size="sm" className="ml-0 sm:ml-2 mt-2 sm:mt-0 shadow-lg" style={{background: '#0B674E', color: '#fff'}} onClick={handleSaveUserData} disabled={saving}>
                    {saving ? t('common.loading', 'Saving...') : t('common.save', 'Save')}
                  </Button>
                )}
                {showHebrewInputWarning && (
                  <div className="text-xs sm:text-sm text-red-500">{t('settings.input_hebrew_warning', 'Please change your input language to Hebrew for all fields except email.')}</div>
                )}
              </>
            ) : (
              <>
                <div><b>{t('settings.f_name')}:</b> {userData.f_name}</div>
                <div><b>{t('settings.s_name')}:</b> {userData.s_name}</div>
                <div><b>{t('settings.p_id')}:</b> {userData.p_id}</div>
                <div><b>{t('settings.email')}:</b> {userData.email}</div>
                <div>
                  <b>{t('settings.phone')}:</b>
                  <Input
                    className="inline-block w-full sm:w-48 ml-0 sm:ml-2 mt-1 sm:mt-0"
                    value={phone}
                    onChange={e => {
                      setPhone(e.target.value);
                      setPhoneChanged(e.target.value !== (userData?.phone || ''));
                    }}
                    disabled={userData.role_id !== 0}
                    lang={isHebrew ? 'he' : undefined}
                    dir={isHebrew ? 'rtl' : 'ltr'}
                  />
                  {phoneChanged && (
                    <Button size="sm" className="ml-0 sm:ml-2 mt-2 sm:mt-0 bg-[#1899D3] hover:bg-[#1479A3] text-white shadow-lg" onClick={handleSavePhone} disabled={saving}>
                      {saving ? t('common.loading', 'Saving...') : t('common.save', 'Save')}
                    </Button>
                  )}
                </div>
                {showHebrewInputWarning && (
                  <div className="text-xs sm:text-sm text-red-500">{t('settings.input_hebrew_warning', 'Please change your input language to Hebrew for all fields except email.')}</div>
                )}
              </>
            )}
          </div>
        )}
        {isAdmin && userData && (
          <div className="mb-4">
            <div className="font-medium mb-2">{t('paymentMethods.title', 'Доступные способы оплаты')}</div>
            <div className="flex gap-4 mb-2">
              {PAYMENT_METHODS.map(method => (
                <label key={method.value} className="flex items-center gap-1">
                  <input
                    type="checkbox"
                    checked={paymentMethods.includes(method.value)}
                    onChange={e => {
                      if (e.target.checked) {
                        setPaymentMethods(prev => [...prev, method.value]);
                      } else {
                        setPaymentMethods(prev => prev.filter(v => v !== method.value));
                      }
                    }}
                  />
                  {method.label}
                </label>
              ))}
            </div>
            <Button size="sm" onClick={handleSavePaymentMethods} disabled={saving} className="shadow-lg">
              {t('paymentMethods.save', 'Сохранить способы оплаты')}
            </Button>
          </div>
        )}
        {selectedUser && (
          <div>
            <div className="font-medium mb-2 text-sm sm:text-base">{t('devices.device', 'Device')}</div>
            {/* Показываем устройства для всех user_id */}
            {Object.keys(userDevicesByUserId).length > 0 && (
              <div className="space-y-4">
                {allUserIds.map(uid => (
                  <ul key={uid} className="space-y-2">
                    {(userDevicesByUserId[uid] || [])
                      .filter(device => device.device_id)
                      .map(device => (
                        <li key={`${uid}_${device.device_id}`} className="flex items-center justify-between border rounded p-2">
                          <span>{device.device_id} ({getDeviceTypeLabel(device.device_type)})</span>
                        </li>
                      ))}
                  </ul>
                ))}
              </div>
            )}
            {/* Добавление устройства только для выбранного пользователя */}
            {adminMode && (
              <div className="mt-4">
                <label className="block mb-2 font-medium text-sm sm:text-base">{t('devices.device', 'Device')}</label>
                <Select value={selectedDeviceToAdd} onValueChange={setSelectedDeviceToAdd}>
                  <SelectTrigger className="w-full sm:w-48 shadow-lg">
                    <SelectValue placeholder={t('devices.device', 'Device')} />
                  </SelectTrigger>
                  <SelectContent className="shadow-lg">
                    {allDevices
                      .filter(d => d.device_name)
                      .filter(d => !Object.values(userDevicesByUserId).flat().some(ud => ud.device_id === d.device_name))
                      .map((d, idx) => (
                        <SelectItem key={`${d.device_name}_${idx}`} value={d.device_name}>
                          {d.device_name} ({getDeviceTypeLabel(d.device_type)})
                        </SelectItem>
                      ))}
                  </SelectContent>
                </Select>
                <Button
                  className="mt-2 shadow-lg"
                  style={{background: '#0B674E', color: '#fff'}}
                  onClick={handleAddDevice}
                  disabled={!selectedDeviceToAdd || deviceActionLoading}
                >
                  {t('common.add', 'Add')}
                </Button>
                {addDeviceError && <div className="text-red-500 text-sm mt-2">{addDeviceError}</div>}
              </div>
            )}
          </div>
        )}
    
        {/* Family members for selected user (роль 1) */}
        {userRoleId === 1 && (
          <div className="mb-24 sm:mb-56">
            <div className="font-medium mb-2 text-sm sm:text-base">{t('settings.family')}</div>
            <div className="border rounded p-2 sm:p-3">
              {relatedUserInfo && (
                <div className="mb-2 text-xs sm:text-sm">
                  <b>{t('settings.related_user', 'Связанный пользователь')}:</b> {relatedUserInfo.uname || relatedUserInfo.p_id} — {String(t(`relationship.${relatedUserInfo.relationship_type}`, relatedUserInfo.relationship_type || ''))}
                </div>
              )}
              {family.length === 0 ? (
                <div className="text-muted-foreground text-xs sm:text-sm">{t('settings.no_family')}</div>
              ) : (
                <ul className="space-y-2">
                  {family.map(member => (
                    <li key={member.id ?? `${member.p_id}_${member.c_id}`} className="flex flex-col sm:flex-row sm:items-center sm:justify-between space-y-1 sm:space-y-0">
                      <span className="text-sm sm:text-base"><b>{member.c_name}</b> ({t('settings.id_label')}: {member.c_id})</span>
                      <span className="text-xs sm:text-sm">
                        {t('settings.sex_label')}: {t(`settings.sex_${member.c_sex?.toLowerCase?.()}`, member.c_sex)}, {t('settings.age_label')}: {member.c_age}
                      </span>
                      <Button size="sm" variant="outline" className="w-full sm:w-auto border-[#1899D3] text-[#1899D3] hover:bg-[#1899D3] hover:text-white shadow-lg" onClick={() => openEditFamilyModal(member)}>{t('settings.edit')}</Button>
                    </li>
                  ))}
                </ul>
              )}
              
            </div>
          </div>
        )}
        {/* Add user to device (manager) */}
        <div>
          <div className="font-medium mb-2 text-sm sm:text-base">{t('settings.add_user')}</div>
          <Button onClick={() => setShowAddUser(true)} className="w-full sm:w-auto bg-[#1899D3] hover:bg-[#1479A3] text-white shadow-lg">{t('settings.add_user_btn')}</Button>
          {showAddUser && (
            <form onSubmit={handleAddUser} className="mt-2 space-y-2 border rounded p-2 sm:p-3">
              {/* Список устройств выбранного пользователя с чекбоксами */}
              {selectableDeviceObjects.length > 0 && (
                <div className="pt-1">
                  <div className="font-medium text-xs sm:text-sm mb-1">{t('devices.device', 'Device')}</div>
                  <div className="grid grid-cols-1 sm:grid-cols-2 gap-2">
                    {selectableDeviceObjects.map((dev) => {
                      const devId = dev.device_id;
                      const checked = selectedDeviceIdsForNewUser.includes(devId);
                      return (
                        <label key={`mgr_newuser_dev_${devId}`} className="flex items-center gap-2 border rounded p-2">
                          <input
                            type="checkbox"
                            checked={checked}
                            onChange={(e) => {
                              setSelectedDeviceIdsForNewUser(prev => {
                                if (e.target.checked) return Array.from(new Set([...prev, devId]));
                                return prev.filter(id => id !== devId);
                              });
                            }}
                          />
                          <span>{devId}{dev.device_type ? ` (${getDeviceTypeLabel(dev.device_type)})` : ''}</span>
                        </label>
                      );
                    })}
                  </div>
                  <div className="text-xs text-muted-foreground mt-1">{t('devices.device', 'Device')}: {t('common.select', 'Select')} 1+</div>
                </div>
              )}
              <Input 
                placeholder={t('settings.f_name')} 
                value={newUserForm.f_name}
                onChange={(e) => setNewUserForm({...newUserForm, f_name: e.target.value})}
                lang={isHebrew ? 'he' : undefined} 
                dir={isHebrew ? 'rtl' : 'ltr'} 
              />
              <Input 
                placeholder={t('settings.s_name')} 
                value={newUserForm.s_name}
                onChange={(e) => setNewUserForm({...newUserForm, s_name: e.target.value})}
                lang={isHebrew ? 'he' : undefined} 
                dir={isHebrew ? 'rtl' : 'ltr'} 
              />
              <Input 
                placeholder={t('settings.p_id')} 
                value={newUserForm.p_id}
                onChange={(e) => setNewUserForm({...newUserForm, p_id: e.target.value})}
                lang={isHebrew ? 'he' : undefined} 
                dir={isHebrew ? 'rtl' : 'ltr'} 
                required
              />
              <Input 
                placeholder={t('settings.phone')} 
                value={newUserForm.phone}
                onChange={(e) => setNewUserForm({...newUserForm, phone: e.target.value})}
                lang={isHebrew ? 'he' : undefined} 
                dir={isHebrew ? 'rtl' : 'ltr'} 
              />
              <Input 
                placeholder={t('settings.email')} 
                type="email"
                value={newUserForm.email}
                onChange={(e) => setNewUserForm({...newUserForm, email: e.target.value})}
              />
              <Select value={newUserForm.role_id} onValueChange={(value) => setNewUserForm({...newUserForm, role_id: value})}>
                <SelectTrigger className="w-full sm:w-48">
                  <SelectValue placeholder={t('settings.role')} />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="2">{t('settings.role2')}</SelectItem>
                  <SelectItem value="3">{t('settings.role3')}</SelectItem>
                </SelectContent>
              </Select>
              
              <Select value={newUserForm.relationship} onValueChange={(value) => setNewUserForm({...newUserForm, relationship: value})}>
                <SelectTrigger className="w-full sm:w-48">
                  <SelectValue placeholder={t('settings.relationship')} />
                </SelectTrigger>
                <SelectContent>
                  {relationshipTypes.map(relType => (
                    <SelectItem key={relType.relationship_name} value={relType.relationship_name}>
                      {String(t(`relationship.${relType.relationship_name}`, relType.relationship_name))}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <div className="flex gap-2">
                <Button type="submit" disabled={addingUser} className="flex-1 bg-[#1899D3] hover:bg-[#1479A3] text-white shadow-lg">
                  {addingUser ? t('common.loading', 'Сохранение...') : t('settings.save')}
                </Button>
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => setShowAddUser(false)}
                  className="flex-1"
                >
                  {t('common.cancel', 'Отмена')}
                </Button>
              </div>
              {addUserError && (
                <div className="text-red-500 text-sm">{addUserError}</div>
              )}
              {showHebrewInputWarning && (
                <div className="text-xs sm:text-sm text-red-500">{t('settings.input_hebrew_warning', 'Please change your input language to Hebrew for all fields except email.')}</div>
              )}
            </form>
          )}
        </div>
      </CardContent>
      <Dialog open={editFamilyModalOpen} onOpenChange={setEditFamilyModalOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle><center>{t('settings.edit_family_member', 'Редактировать члена семьи')}</center></DialogTitle>
          </DialogHeader>
          {editingFamilyData && (
            <form className="space-y-4" onSubmit={e => { e.preventDefault(); handleSaveFamilyMember(); }}>
              <div>
                <label className="block mb-1 font-medium">{t('settings.f_name', 'Имя')}:</label>
                <Input className={`shadow-lg ${editingFamilyData.c_name ? 'ring-2 ring-[#0B674E] border-[#0B674E]' : ''}`} value={editingFamilyData.c_name} onChange={e => setEditingFamilyData((d: any) => ({ ...d, c_name: e.target.value }))} required />
              </div>
              <div>
                <label className="block mb-1 font-medium">{t('settings.sex_label', 'Пол')}</label>
                <Select value={editingFamilyData.c_sex || ''} onValueChange={v => setEditingFamilyData((d: any) => ({ ...d, c_sex: v }))}>
                  <SelectTrigger className={`w-full shadow-lg ${editingFamilyData.c_sex ? 'ring-2 ring-[#0B674E] border-[#0B674E]' : ''}`} dir={isRTL ? 'rtl' : 'ltr'}>
                    <SelectValue placeholder={t('settings.sex_placeholder', 'Выберите пол')} />
                  </SelectTrigger>
                  <SelectContent dir={isRTL ? 'rtl' : 'ltr'}>
                    <SelectItem value="male" className={isRTL ? 'text-right' : 'text-left'}>{t('settings.sex_male', 'Мужской')}</SelectItem>
                    <SelectItem value="female" className={isRTL ? 'text-right' : 'text-left'}>{t('settings.sex_female', 'Женский')}</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div>
                <label className="block mb-1 font-medium">{t('settings.age_label', 'Возраст')}</label>
                <Input className={`shadow-lg ${editingFamilyData.c_age ? 'ring-2 ring-[#0B674E] border-[#0B674E]' : ''}`} type="number" min={0} value={editingFamilyData.c_age} onChange={e => setEditingFamilyData((d: any) => ({ ...d, c_age: e.target.value }))} required />
              </div>
              <DialogFooter>
                <Button type="submit" className="mt-2 w-full sm:w-auto bg-[#0B674E] hover:bg-[#A8CC8C] text-white shadow-lg" disabled={savingFamily}>{savingFamily ? t('common.loading', 'Сохранение...') : t('common.save', 'Сохранить')}</Button>
              </DialogFooter>
            </form>
          )}
        </DialogContent>
      </Dialog>
    </Card>
  );
};

export default UserSettingsTab; 