import React, { useEffect, useState } from 'react';
import { useTranslation } from 'react-i18next';
import {
  Card,
  CardContent,
  CardHeader,
  CardTitle,
  CardDescription,
} from "../components/ui/card";
import { CreditCard } from "lucide-react";
import { DropdownMenu, DropdownMenuTrigger, DropdownMenuContent, DropdownMenuItem } from '../components/ui/dropdown-menu';
import { format } from 'date-fns';

const API_KEY = process.env.NEXT_PUBLIC_SUMIT_TOKEN;
const COMPANY_ID = process.env.NEXT_PUBLIC_SUMIT_COMPANY_ID;

interface UserData {
  device_id: string;
  role_id: number;
  payment_history_id: string;
  email: string;
  f_name?: string;
  s_name?: string;
  p_id?: string; // Added p_id to UserData interface
}

interface ItemRow {
  DocumentID: string;
  Date: string;
  Name: string;
  Quantity: number;
  UnitPrice: number;
  TotalPrice: number;
}

const PaymentHistoryTab: React.FC = () => {
  const { t, i18n } = useTranslation();
  const isRTL = i18n.dir() === 'rtl';
  const [users, setUsers] = useState<UserData[]>([]);
  const [selectedUser, setSelectedUser] = useState<string>('');
  const [paymentHistoryUrl, setPaymentHistoryUrl] = useState<string>('');
  const [searchUser, setSearchUser] = useState('');
  const [dateFrom, setDateFrom] = useState<string>('2023-01-01');
  const [dateTo, setDateTo] = useState<string>(format(new Date(), 'yyyy-MM-dd'));
  const [items, setItems] = useState<ItemRow[]>([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [openDocId, setOpenDocId] = useState<string | null>(null);
  const [docDetails, setDocDetails] = useState<Record<string, { items: any[], downloadUrl?: string }>>({});


  useEffect(() => {
    fetch('/api/postgres/users')
      .then(res => res.json())
      .then(data => setUsers(data.users || []));
  }, []);

  useEffect(() => {
    if (!selectedUser) {
      setPaymentHistoryUrl('');
      return;
    }
    const user = users.find(u => u.email === selectedUser);
    if (user?.payment_history_id) {
      setPaymentHistoryUrl(`https://pay.sumit.co.il/6uhbev/a/history/${user.payment_history_id}`);
    } else {
      setPaymentHistoryUrl('');
    }
  }, [selectedUser, users]);

  // Получить p_id выбранного пользователя
  const selectedUserObj = users.find(u => u.email === selectedUser);
  const customerName = selectedUserObj?.p_id;

  // Загрузка данных при изменении пользователя или периода
  useEffect(() => {
    if (!customerName || !dateFrom || !dateTo) {
      setItems([]);
      return;
    }
    setLoading(true);
    setError(null);

    async function fetchAllDocuments() {
      let allDocs: any[] = [];
      let page = 0;
      const pageSize = 100;
      while (true) {
        const resp = await fetch('https://api.sumit.co.il/accounting/documents/list/', {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            Accept: 'application/json',
          },
          body: JSON.stringify({
            Credentials: {
              CompanyID: COMPANY_ID,
              APIKey: API_KEY,
            },
            DateFrom: dateFrom + 'T00:00:00Z',
            DateTo: dateTo + 'T23:59:59Z',
            IncludeDrafts: false,
            Paging: {
              StartIndex: page * pageSize,
              PageSize: pageSize,
            },
          }),
        });
        const json = await resp.json();
        const docs = json?.Data?.Documents || [];
        allDocs = allDocs.concat(docs);
        if (docs.length < pageSize) break;
        page++;
      }
      return allDocs;
    }

    async function fetchData() {
      try {
        const allDocs = (await fetchAllDocuments()) || [];
        const docs = allDocs.filter((doc: any) => doc.CustomerName === customerName);
        const allItems: ItemRow[] = [];
        for (const doc of docs) {
          const res = await fetch('https://api.sumit.co.il/accounting/documents/getdetails/', {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              Accept: 'application/json',
            },
            body: JSON.stringify({
              Credentials: {
                CompanyID: COMPANY_ID,
                APIKey: API_KEY,
              },
              DocumentID: doc.DocumentID,
              DocumentType: "0",
            }),
          });
          const details = await res.json();
          const itemsArr = details?.Data?.Items || [];
          const docDate = details?.Data?.Document?.Date || '';
          for (const item of itemsArr) {
            allItems.push({
              DocumentID: doc.DocumentID,
              Date: docDate,
              Name: item.Item?.Name || '',
              Quantity: item.Quantity,
              UnitPrice: item.UnitPrice,
              TotalPrice: item.TotalPrice,
            });
          }
        }
        setItems(allItems);
      } catch (e: any) {
        setError(e.message || 'Ошибка загрузки данных');
        setItems([]);
      }
      setLoading(false);
    }
    fetchData();
  }, [customerName, dateFrom, dateTo]);

  // Группировка по DocumentID
  const groupedDocs = items.reduce<Record<string, { date: string, total: number, items: ItemRow[] }>>((acc, item) => {
    if (!acc[item.DocumentID]) {
      acc[item.DocumentID] = { date: item.Date, total: 0, items: [] };
    }
    acc[item.DocumentID].total += item.TotalPrice;
    acc[item.DocumentID].items.push(item);
    return acc;
  }, {});
  const docIds = Object.keys(groupedDocs);

  // Функция для загрузки деталей документа (если не загружено)
  async function handleOpenDoc(docId: string) {
    if (docDetails[docId]) {
      setOpenDocId(openDocId === docId ? null : docId);
      return;
    }
    setLoading(true);
    try {
      const res = await fetch('https://api.sumit.co.il/accounting/documents/getdetails/', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          Accept: 'application/json',
        },
        body: JSON.stringify({
          Credentials: {
            CompanyID: COMPANY_ID,
            APIKey: API_KEY,
          },
          DocumentID: docId,
          DocumentType: "0",
        }),
      });
      const details = await res.json();
      const itemsArr = details?.Data?.Items || [];
      const downloadUrl = details?.Data?.Document?.DocumentDownloadURL || details?.DocumentDownloadURL || '';
      setDocDetails(prev => ({ ...prev, [docId]: { items: itemsArr, downloadUrl } }));
      setOpenDocId(docId);
    } catch (e) {
      setError(t('paymentHistory.detailsError', 'Failed to load document details'));
    }
    setLoading(false);
  }


  return (
    <Card className="w-full max-w-2xl mb-40 mx-auto sm:max-w-full shadow-lg">
      <CardHeader>
        <CardTitle className="flex items-center text-lg sm:text-base">
          {t('paymentHistory.title', 'Payment History')}
        </CardTitle>
        <CardDescription className="sm:text-xs">{t('paymentHistory.description', 'View your payment history transactions below. This content is loaded from an external site.')}</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="mb-4 flex flex-col sm:flex-row gap-2 items-center">
          {/* Выбор пользователя */}
          <div className="flex-1">
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <button
                  type="button"
                  className={`w-full border rounded p-2 mb-2 bg-white ${isRTL ? 'text-right' : 'text-left'}`}
                  style={{ direction: isRTL ? 'rtl' : 'ltr', textAlign: isRTL ? 'right' : 'left' }}
                >
                  {selectedUser
                    ? (users.find(u => u.email === selectedUser)?.f_name || '') +
                      (users.find(u => u.email === selectedUser)?.s_name ? ' ' + users.find(u => u.email === selectedUser)?.s_name : '')
                    : t('systemAdminTab.allUsers', 'All Users')}
                </button>
              </DropdownMenuTrigger>
              <DropdownMenuContent className="max-h-60 overflow-y-auto shadow-lg">
                <input
                  type="text"
                  placeholder={t('systemAdminTab.searchUser', 'Search user...')}
                  className="w-full border rounded p-2 mb-2"
                  value={searchUser || ''}
                  onChange={e => setSearchUser(e.target.value)}
                  dir={isRTL ? 'rtl' : 'ltr'}
                  style={{ textAlign: isRTL ? 'right' : 'left' }}
                />
                <DropdownMenuItem
                  onSelect={() => setSelectedUser('')}
                  className={selectedUser === '' ? `bg-accent text-accent-foreground ${isRTL ? 'text-right' : 'text-left'}` : isRTL ? 'text-right' : 'text-left'}
                  style={{ direction: isRTL ? 'rtl' : 'ltr', textAlign: isRTL ? 'right' : 'left' }}
                >
                  {t('systemAdminTab.allUsers', 'All Users')}
                </DropdownMenuItem>
                {Array.from(new Map(users.map(u => [u.email, u])).values())
                  .filter(u => {
                    const search = (searchUser || '').toLowerCase();
                    return (
                      (u.f_name && u.f_name.toLowerCase().includes(search)) ||
                      (u.s_name && u.s_name.toLowerCase().includes(search))
                    );
                  })
                  .map((u, idx) => (
                    <DropdownMenuItem
                      key={u.email + '_' + (u.payment_history_id || idx)}
                      onSelect={() => setSelectedUser(u.email)}
                      className={selectedUser === u.email ? `bg-accent text-accent-foreground ${isRTL ? 'text-right' : 'text-left'}` : isRTL ? 'text-right' : 'text-left'}
                      style={{ direction: isRTL ? 'rtl' : 'ltr', textAlign: isRTL ? 'right' : 'left', background: selectedUser === u.email ? '#a3d9f2' : undefined }}
                    >
                      {(u.f_name || '') + (u.s_name ? ' ' + u.s_name : '') || u.email}
                    </DropdownMenuItem>
                  ))}
              </DropdownMenuContent>
            </DropdownMenu>
          </div>
          {/* Выбор периода */}
          <div className="flex gap-2 items-center">
            <label className="font-medium text-sm">{t('paymentHistory.dateFrom', 'From')}</label>
            <input type="date" value={dateFrom} onChange={e => setDateFrom(e.target.value)} className="border rounded p-1" />
            <label className="font-medium text-sm">{t('paymentHistory.dateTo', 'To')}</label>
            <input type="date" value={dateTo} onChange={e => setDateTo(e.target.value)} className="border rounded p-1" />
          </div>
        </div>
        {/* Таблица с результатами */}
        {loading ? (
          <p>{t('paymentHistory.loading', 'Loading payment history...')}</p>
        ) : error ? (
          <p className="text-red-500">{error}</p>
        ) : docIds.length > 0 ? (
          <div className="overflow-x-auto">
            <table className="min-w-full border text-sm">
              <thead>
                <tr style={{ background: '#e5e5e5' }}>
                  <th className="border px-2 py-1">{t('paymentHistory.documentId', 'Document ID')}</th>
                  <th className="border px-2 py-1">{t('paymentHistory.date', 'Date')}</th>
                  <th className="border px-2 py-1">{t('paymentHistory.totalPrice', 'Total Price')}</th>
                </tr>
              </thead>
              <tbody>
                {docIds.map(docId => (
                  <React.Fragment key={docId}>
                    <tr
                      className="cursor-pointer hover:bg-accent"
                      onClick={() => handleOpenDoc(docId)}
                      style={{ background: openDocId === docId ? '#f0f4fa' : undefined }}
                    >
                      <td className="border px-2 py-1">{docId}</td>
                      <td className="border px-2 py-1">{groupedDocs[docId].date ? format(new Date(groupedDocs[docId].date), 'yyyy-MM-dd') : ''}</td>
                      <td className="border px-2 py-1">{groupedDocs[docId].total.toFixed(2)}</td>
                    </tr>
                    {openDocId === docId && docDetails[docId] && (
                      <tr>
                        <td colSpan={3} className="p-0">
                          <div className="p-2 border-t bg-white">
                            <div className="mb-2 font-medium">{t('paymentHistory.items', 'Items')}</div>
                            <table className="min-w-full border text-xs mb-2">
                              <thead>
                                <tr style={{ background: '#e5e5e5' }}>
                                  <th className="border px-2 py-1">{t('paymentHistory.item', 'Item')}</th>
                                  <th className="border px-2 py-1">{t('paymentHistory.quantity', 'Quantity')}</th>
                                  <th className="border px-2 py-1">{t('paymentHistory.unitPrice', 'Unit Price')}</th>
                                  <th className="border px-2 py-1">{t('paymentHistory.totalPrice', 'Total Price')}</th>
                                </tr>
                              </thead>
                              <tbody>
                                {docDetails[docId].items.map((item, idx) => (
                                  <tr key={idx}>
                                    <td className="border px-2 py-1">{item.Name || (item.Item && item.Item.Name) || ''}</td>
                                    <td className="border px-2 py-1">{item.Quantity}</td>
                                    <td className="border px-2 py-1">{item.UnitPrice}</td>
                                    <td className="border px-2 py-1">{item.TotalPrice}</td>
                                  </tr>
                                ))}
                              </tbody>
                            </table>
                            {docDetails[docId].downloadUrl && (
                              <a
                                href={docDetails[docId].downloadUrl}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="inline-block px-4 py-2 bg-[#0B674E] text-white rounded hover:bg-[#09553d]"
                              >
                                {t('paymentHistory.download', 'Download document')}
                              </a>
                            )}
                          </div>
                        </td>
                      </tr>
                    )}
                  </React.Fragment>
                ))}
              </tbody>
            </table>
          </div>
        ) : selectedUser ? (
          <p>{t('paymentHistory.noData', 'No payment data found for this period.')}</p>
        ) : null}
      </CardContent>
    </Card>
  );
};

export default PaymentHistoryTab; 