import React, { useState, useEffect } from 'react';
import { useTranslation } from 'react-i18next';
import { Card, CardHeader, CardTitle, CardContent } from '../components/ui/card';
import { Button } from '../components/ui/button';
import { DropdownMenu, DropdownMenuTrigger, DropdownMenuContent, DropdownMenuItem } from '../components/ui/dropdown-menu';
import { Alert, AlertTitle, AlertDescription } from '../components/ui/alert';
import { AlertCircle } from 'lucide-react';

const ManagerPushTab: React.FC = () => {
  const { t, i18n } = useTranslation();
  const isRTL = i18n.dir() === 'rtl';
  const [users, setUsers] = useState<{ email: string; f_name?: string; s_name?: string }[]>([]);
  const [selectedUser, setSelectedUser] = useState<string>('all');
  const [notifTitle, setNotifTitle] = useState('');
  const [notifMessage, setNotifMessage] = useState('');
  const [notifLoading, setNotifLoading] = useState(false);
  const [notifError, setNotifError] = useState<string | null>(null);
  const [notifSuccess, setNotifSuccess] = useState<string | null>(null);
  const [searchUser, setSearchUser] = useState('');

  useEffect(() => {
    fetch('/api/postgres/users')
      .then(res => res.json())
      .then(data => setUsers(data.users || []));
  }, []);

  const uniqueUsers = Array.from(new Map(users.map(u => [u.email, u])).values());

  const handleSendNotification = async (e: React.FormEvent) => {
    e.preventDefault();
    setNotifLoading(true);
    setNotifError(null);
    setNotifSuccess(null);
    try {
      const res = await fetch('/api/push/send', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          to: selectedUser,
          title: notifTitle,
          message: notifMessage,
        })
      });
      if (!res.ok) {
        const errData = await res.json();
        throw new Error(errData.error || t('managerPanel.failedToSendNotification', 'Не удалось отправить уведомление'));
      }
      setNotifSuccess(t('managerPanel.notificationSent', 'Уведомление успешно отправлено!'));
      setNotifTitle('');
      setNotifMessage('');
      setSelectedUser('all');
    } catch (err: any) {
      setNotifError(err.message || t('managerPanel.failedToSendNotification', 'Не удалось отправить уведомление'));
    } finally {
      setNotifLoading(false);
    }
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle className={isRTL ? 'text-right' : ''}>{t('managerPanel.sendPushNotification', 'Отправить push-уведомление')}</CardTitle>
      </CardHeader>
      <CardContent>
        <form onSubmit={handleSendNotification} className="space-y-4 max-w-md w-full mx-auto shadow-lg p-4 rounded-lg bg-white">
          <div>
            <label className={`block mb-1 font-medium ${isRTL ? 'text-right' : ''}`}>{t('managerPanel.sendTo', 'Кому отправить')}</label>
            <DropdownMenu>
              <DropdownMenuTrigger asChild>
                <button
                  type="button"
                  className={`w-full border rounded p-2 mb-2 bg-white ${isRTL ? 'text-right' : 'text-left'}`}
                  style={{ direction: isRTL ? 'rtl' : 'ltr', textAlign: isRTL ? 'right' : 'left' }}
                >
                  {selectedUser === 'all'
                    ? t('managerPanel.allUsers', 'Все пользователи')
                    : (
                        uniqueUsers.find(u => u.email === selectedUser)?.f_name || ''
                      ) +
                      (uniqueUsers.find(u => u.email === selectedUser)?.s_name
                        ? ' ' + uniqueUsers.find(u => u.email === selectedUser)?.s_name
                        : '') || selectedUser}
                </button>
              </DropdownMenuTrigger>
              <DropdownMenuContent className="max-h-60 overflow-y-auto shadow-lg">
                <input
                  type="text"
                  placeholder={t('managerPanel.searchUser', 'Поиск пользователя...')}
                  className="w-full border rounded p-2 mb-2 shadow-lg"
                  value={searchUser || ''}
                  onChange={e => setSearchUser(e.target.value)}
                  dir={isRTL ? 'rtl' : 'ltr'}
                  style={{ textAlign: isRTL ? 'right' : 'left' }}
                />
                <DropdownMenuItem
                  onSelect={() => setSelectedUser('all')}
                  className={selectedUser === 'all' ? `bg-accent text-accent-foreground ${isRTL ? 'text-right' : 'text-left'}` : isRTL ? 'text-right' : 'text-left'}
                  style={{ direction: isRTL ? 'rtl' : 'ltr', textAlign: isRTL ? 'right' : 'left' }}
                >
                  {t('managerPanel.allUsers', 'Все пользователи')}
                </DropdownMenuItem>
                {uniqueUsers
                  .filter(u => {
                    const search = (searchUser || '').toLowerCase();
                    return (
                      (u.f_name && u.f_name.toLowerCase().includes(search)) ||
                      (u.s_name && u.s_name.toLowerCase().includes(search))
                    );
                  })
                  .map(u => (
                    <DropdownMenuItem
                      key={u.email}
                      onSelect={() => setSelectedUser(u.email)}
                      className={selectedUser === u.email ? `bg-accent text-accent-foreground ${isRTL ? 'text-right' : 'text-left'}` : isRTL ? 'text-right' : 'text-left'}
                      style={{ direction: isRTL ? 'rtl' : 'ltr', textAlign: isRTL ? 'right' : 'left' }}
                    >
                      {(u.f_name || '') + (u.s_name ? ' ' + u.s_name : '') || u.email}
                    </DropdownMenuItem>
                  ))}
              </DropdownMenuContent>
            </DropdownMenu>
          </div>
          <div>
            <label className={`block mb-1 font-medium ${isRTL ? 'text-right' : ''}`}>{t('managerPanel.title', 'Заголовок')}</label>
            <input
              className="w-full border rounded p-2 shadow-lg"
              value={notifTitle}
              onChange={e => setNotifTitle(e.target.value)}
              required
              dir={isRTL ? 'rtl' : 'ltr'}
            />
          </div>
          <div>
            <label className={`block mb-1 font-medium ${isRTL ? 'text-right' : ''}`}>{t('managerPanel.message', 'Сообщение')}</label>
            <textarea
              className="w-full border rounded p-2 shadow-lg"
              value={notifMessage}
              onChange={e => setNotifMessage(e.target.value)}
              required
              dir={isRTL ? 'rtl' : 'ltr'}
            />
          </div>
          <Button type="submit" disabled={notifLoading} className="w-full shadow-lg" style={{ background: '#0B674E', color: '#fff' }}>
            {notifLoading ? t('managerPanel.sending', 'Отправка...') : t('managerPanel.sendNotification', 'Отправить уведомление')}
          </Button>
          {notifError && (
            <Alert variant="destructive">
              <AlertCircle className="h-4 w-4" />
              <AlertTitle>{t('common.error', 'Ошибка')}</AlertTitle>
              <AlertDescription>{notifError}</AlertDescription>
            </Alert>
          )}
          {notifSuccess && (
            <Alert>
              <AlertTitle>{t('common.success', 'Успех')}</AlertTitle>
              <AlertDescription>{notifSuccess}</AlertDescription>
            </Alert>
          )}
        </form>
      </CardContent>
    </Card>
  );
};

export default ManagerPushTab; 