"use client";

import React, { useEffect, useState, useMemo } from "react";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
  DialogClose,
  DialogTrigger,
} from "../components/ui/dialog";
import { Button } from "../components/ui/button";
import { Alert, AlertDescription, AlertTitle } from "../components/ui/alert";
import { AlertCircle } from "lucide-react";
import { Skeleton } from "../components/ui/skeleton";
import { Input } from "../components/ui/input";
import { Label } from "../components/ui/label";
import { loadMqttLibrary, connectAndSend, MQTT_COMMANDS } from "../services/mqttService";
import GaugeChart from "./GaugeChart";
import { useAuth } from '../hooks/use-auth';
import { useTranslation } from 'react-i18next';

// Mirroring DeviceData structure from SystemDashboardTab for the device prop
interface DeviceData {
  deviceId: string;
  maxValue: number | null;
  // Data for main dashboard charts (based on selectedDuration)
  chartCurrentReading?: number | null;
  chartValveState?: string | null;
  // Data for table & modal gauge (last 24h)
  twentyFourHourCurrentReading?: number | null;
  twentyFourHourValveState?: string | null;
  // monthlyData, roleId, isAdminDevice are not directly used by this modal's gauge logic,
  // but keeping them if the parent passes the full DeviceData object.
  monthlyData?: { month: string; usage: number }[];
  roleId: number;
  isAdminDevice: boolean;
  device_type?: string;
  dev_eui?: string;
}

interface DeviceDetailModalProps {
  isOpen: boolean;
  onClose: () => void;
  device: DeviceData | null;
  onMaxValueUpdate?: (deviceId: string, newMaxValue: number) => void;
}

interface MqttStatus {
  message: string;
  type: "error" | "success" | "info";
}

// MQTT Configuration (global or component-level, ensure mqtt is typed)
declare var mqtt: any; // For MQTT.js loaded via script

const MQTT_BROKER_URL = process.env.NEXT_PUBLIC_MQTT_BROKER_URL;
const MQTT_APP_ID = process.env.NEXT_PUBLIC_MQTT_APP_ID;

// Commands from user example
const FPORT = 2;

let mqttClientInstance: any = null; // mqtt.MqttClient
let alertsShown: { [key: string]: boolean } = {};
Object.keys(MQTT_COMMANDS).forEach((key) => {
  if (key.includes("%")) alertsShown[key] = false;
});
alertsShown["addMore"] = false;
alertsShown["meterClosed"] = false;

// Add valve commands to MQTT_COMMANDS
const VALVE_COMMANDS = {
  "OPEN_VALVE": "Jh8ARQ==",
  "CLOSE_VALVE": "Jh8BRg=="
};

const DeviceDetailModal: React.FC<DeviceDetailModalProps> = ({
  isOpen,
  onClose,
  device,
  onMaxValueUpdate,
}) => {
  const { userPermissions, user } = useAuth();
  const { t } = useTranslation();
  const [isLoadingGauge, setIsLoadingGauge] = useState(true);
  const [addMaxValue, setAddMaxValue] = useState<string>("");
  const [currentDeviceMaxValue, setCurrentDeviceMaxValue] = useState<number | null>(device?.maxValue ?? null);
  const [isUpdating, setIsUpdating] = useState(false);
  const [mqttStatus, setMqttStatus] = useState<MqttStatus | null>(null);
  const [alertMessage, setAlertMessage] = useState<string>("");
  const [alertType, setAlertType] = useState<"error" | "success" | "info">("info");
  const [localDevice, setLocalDevice] = useState<DeviceData | null>(null);
  const [mqttLoaded, setMqttLoaded] = useState(false);
  const [gaugeReloadKey, setGaugeReloadKey] = useState(0);

  const devEUI = useMemo(() => device?.dev_eui ? device.dev_eui : (device ? `8254812310000${device.deviceId.slice(-3)}` : null), [device]);
  const deviceName = useMemo(() => device?.deviceId, [device]); // This is the deviceId from your system

  // Update currentDeviceMaxValue when device changes
  useEffect(() => {
    if (device?.maxValue !== undefined) {
      setCurrentDeviceMaxValue(Number(device.maxValue));
      if (localDevice) {
        setLocalDevice({
          ...localDevice,
          maxValue: Number(device.maxValue)
        });
      }
    }
  }, [device?.maxValue]);

  // Update localDevice when device prop changes
  useEffect(() => {
    if (device) {
      setLocalDevice(device);
    }
  }, [device]);
  
  // Effect for initializing database and fetching command flags
  useEffect(() => {
    if (!isOpen || !device || !deviceName) return;

    const initDbAndFetchFlags = async () => {
      setIsUpdating(true);
      try {
        // Initialize database columns
        await fetch(`/api/postgres/device-management`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ 
            action: 'initialize_threshold_columns',
          }),
        });

        // Fetch command flags
        const flagsResponse = await fetch(`/api/postgres/device-management`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ 
            action: 'get_command_flags',
            deviceId: deviceName 
          }),
        });

        if (flagsResponse.ok) {
          const flagsData = await flagsResponse.json();
          // setCommandFlags(flagsData.flags); 
        } else {
          setMqttStatus({ 
            message: `Failed to fetch command status for ${deviceName}. Threshold commands might not work as expected.`, 
            type: 'error' 
          });
          // setCommandFlags(null);
        }
      } catch (error: any) {
        setMqttStatus({ 
          message: "An error occurred while setting up device controls. Check console.", 
          type: 'error' 
        });
      } finally {
        setIsUpdating(false);
      }
    };

    initDbAndFetchFlags();
  }, [isOpen, device, deviceName]);

  useEffect(() => {
    loadMqttLibrary()
      .then(() => {
        setMqttLoaded(true);
      })
      .catch((error) => {
        console.error("Failed to load MQTT library:", error);
        setMqttStatus({ message: "Failed to load MQTT library", type: 'error' });
      });
  }, []);

  const handleMaxValueUpdate = async () => {
    if (!deviceName) {
      setAlertMessage("שגיאה: מזהה השעון חסר");
      setAlertType("error");
      return;
    }

    // Parse input as float, always using dot as decimal separator
    const updateValue = Number(addMaxValue.replace(',', '.'));
    if (isNaN(updateValue) || addMaxValue.trim() === '' || updateValue === 0) {
      setAlertMessage("שגיאה: הערך המקסימלי חייב להיות מספר שונה מאפס");
      setAlertType("error");
      return;
    }

    // Calculate new max value by adding to current max value
    const currentMax = Number(currentDeviceMaxValue) ?? 0;
    const newMaxValue = currentMax + updateValue;

    if (isNaN(newMaxValue)) {
      setAlertMessage("שגיאה: חישוב הערך המקסימלי נכשל");
      setAlertType("error");
      return;
    }

    setIsUpdating(true);
    try {
      // Only superuser (role_id=0) can update directly
      if (userPermissions?.can_approve_maximum) {
        const payload = {
          action: 'update_max_value',
          deviceId: deviceName,
          newMaxValue: newMaxValue,
          userPermissions: userPermissions
        };
        //console.log('Sending max value update payload:', payload);
        const response = await fetch('/api/postgres/device-management', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(payload),
        });

        if (!response.ok) {
          const errorData = await response.json();
          throw new Error(errorData.message || 'Failed to update max value');
        }

        const updatedData = await response.json();
        const updatedMaxValue = Number(updatedData.newMaxValue);
        
        if (isNaN(updatedMaxValue)) {
          throw new Error('Invalid max value received from server');
        }

        // Update all relevant state
        setCurrentDeviceMaxValue(updatedMaxValue);
        setAddMaxValue('');
        setAlertMessage(`הערך המקסימלי עודכן בהצלחה: ${currentMax.toFixed(1)} + ${updateValue.toFixed(1)} = ${updatedMaxValue.toFixed(1)}`);
        setAlertType("success");
        setGaugeReloadKey((k) => k + 1);

        // Update the local device state to trigger gauge refresh
        if (localDevice) {
          setLocalDevice({
            ...localDevice,
            maxValue: updatedMaxValue
          });
        }

        // Check if valve is closed and automatically open it
        const currentValveState = device?.twentyFourHourValveState?.toLowerCase().trim();
        if (currentValveState === 'close valve' && userPermissions?.can_open_valve && mqttLoaded) {
          //console.log('Valve is currently closed. Automatically sending OPEN_VALVE command...');
          try {
            await handleValveControl("OPEN_VALVE");
            setAlertMessage(`הערך המקסימלי עודכן בהצלחה: ${currentMax.toFixed(1)} + ${updateValue.toFixed(1)} = ${updatedMaxValue.toFixed(1)}. השסתום נפתח אוטומטית.`);
            setAlertType("success");
          } catch (valveError) {
            console.error('Failed to automatically open valve:', valveError);
            setAlertMessage(`הערך המקסימלי עודכן בהצלחה: ${currentMax.toFixed(1)} + ${updateValue.toFixed(1)} = ${updatedMaxValue.toFixed(1)}. שגיאה בפתיחת השסתום האוטומטית.`);
            setAlertType("info");
          }
        }

        // Notify parent component about the update
        if (onMaxValueUpdate && deviceName) {
          onMaxValueUpdate(deviceName, updatedMaxValue);
        }
      } else {
        // For non-superusers, create a pending approval request
        const response = await fetch('/api/postgres/pending-max-value-requests', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            deviceId: deviceName,
            requestedValue: newMaxValue,
            requestedBy: user?.email || 'Unknown'
          })
        });

        if (!response.ok) {
          throw new Error('Failed to create approval request');
        }

        setAddMaxValue('');
        setAlertMessage('בקשה לעדכון ערך מקסימלי נשלחה לסופר-אדמין וממתינה לאישור. הערך לא שונה בפועל עד לאישור.');
        setAlertType('info');
      }
    } catch (error: any) {
      console.error('Error updating max value:', error);
      setAlertMessage(error.message || "שגיאה בעדכון הערך המקסימלי");
      setAlertType("error");
    } finally {
      setIsUpdating(false);
    }
  };

  const handleValveControl = async (commandType: "OPEN_VALVE" | "CLOSE_VALVE") => {
    if (!devEUI || !deviceName) {
      setMqttStatus({ message: "Device information is missing.", type: 'error' });
      return;
    }
    if (!mqttLoaded) {
      setMqttStatus({ message: "MQTT library not loaded", type: 'error' });
        return;
    }
    // Clear previous MQTT status before sending a new command
    setMqttStatus(null); 
    await connectAndSend(devEUI, VALVE_COMMANDS[commandType]);
  };

  useEffect(() => {
    if (device && typeof device.twentyFourHourCurrentReading === 'number' && device.maxValue !== null) {
      setIsLoadingGauge(false);
    } else if (device) {
      setIsLoadingGauge(false); 
    } else {
      setIsLoadingGauge(true); 
    }
  }, [device]);

  // Add type guard for currentValue
  const getCurrentValue = (): number | null => {
    if (device?.twentyFourHourCurrentReading === undefined) return null;
    return device.twentyFourHourCurrentReading;
  };

  if (!device && isOpen) {
    // This case should ideally not happen if logic in parent is correct, but as a fallback:
    return (
        <Dialog open={isOpen} onOpenChange={onClose}>
            <DialogContent><DialogHeader><DialogTitle>Error</DialogTitle></DialogHeader><p>Device data is missing.</p><DialogFooter><Button onClick={onClose}>Close</Button></DialogFooter></DialogContent>
        </Dialog>
    );
  }
  if (!device) return null; // If not open and no device, render nothing

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogTrigger asChild>
        <Button variant="outline">{t('deviceDetail.deviceDetails', 'Device Details')}</Button>
      </DialogTrigger>
      <DialogContent className="w-[90vw] max-w-lg sm:max-w-xl md:max-w-2xl">
        <DialogTitle className="sr-only">{t('deviceDetail.deviceDetails', 'Device Details')}</DialogTitle>
        {/* Удалён DialogHeader с заголовками */}
        <div className="grid gap-4 py-4">
          {/* Gauge Chart */}
          <GaugeChart
            key={gaugeReloadKey}
            deviceName={deviceName || ''}
            devEUI={devEUI || ''}
            unit={device?.device_type === 'Electric meter' ? 'kWh' : 'm³'}
            percentTextColor="#AAA9A9"
            valveTextColor="#AAA9A9"
          />

          {/* Max Value Update Section */}
          <div className="grid grid-cols-4 items-center gap-4">
            <Input
              id="maxValue"
              type="number"
              value={addMaxValue}
              onChange={(e) => setAddMaxValue(e.target.value)}
              className="col-span-2 shadow-lg"
              placeholder={t('deviceDetail.enterValueToAdd', 'Enter value to add to max')}
            />
            <Button onClick={handleMaxValueUpdate} className="col-span-2 shadow-lg" style={{ background: '#253368', color: '#fff' }}>{t('deviceDetail.updateMaxValue', 'Update Max Value')}</Button>
          </div>

          {/* Valve Control Buttons */}
          <div className="flex gap-2 justify-center">
            {userPermissions?.can_open_valve && (
            <Button 
              onClick={() => handleValveControl("OPEN_VALVE")}
              disabled={isUpdating}
              className="flex-1 max-w-[200px] font-bold"
              style={{ background: '#0B674E', color: '#fff' }}
            >
              {t('deviceDetail.openValve', 'OPEN VALVE')}
            </Button>
            )}
            {userPermissions?.can_close_valve && (
            <Button 
              onClick={() => handleValveControl("CLOSE_VALVE")}
              disabled={isUpdating}
              className="flex-1 max-w-[200px] font-bold"
              style={{ background: '#D74322', color: '#fff' }}
            >
              {t('deviceDetail.closeValve', 'CLOSE VALVE')}
            </Button>
            )}
        </div>

          {/* Status Messages */}
        {mqttStatus && (
          <Alert variant={mqttStatus.type === 'error' ? 'destructive' : 'default'} className="mb-4">
            <AlertCircle className="h-4 w-4" />
            <AlertTitle>{mqttStatus.type === 'error' ? 'Error' : mqttStatus.type === 'success' ? 'Success' : 'Info'}</AlertTitle>
            <AlertDescription>{mqttStatus.message}</AlertDescription>
          </Alert>
        )}
          {alertMessage && (
            <Alert variant={alertType === 'error' ? 'destructive' : 'default'} className="mb-4">
                    <AlertCircle className="h-4 w-4" />
              <AlertTitle>{alertType === 'error' ? 'Error' : alertType === 'success' ? 'Success' : 'Info'}</AlertTitle>
              <AlertDescription>{alertMessage}</AlertDescription>
                </Alert>
            )}
          </div>
      </DialogContent>
    </Dialog>
  );
};

export default DeviceDetailModal; 