import { NextResponse } from 'next/server';
import { postgresClient } from '../../../../lib/postgresql-client';

export async function POST(request: Request) {
  try {
    const { email, allDevices } = await request.json();
    if (!email) {
      return NextResponse.json({ error: 'Email is required' }, { status: 400 });
    }

    let deviceIds: string[] = [];

    if (allDevices) {
      // Get all devices from device_max_value table
      const allDevicesQuery = `SELECT device_name FROM device_max_value`;
      const allDevicesResult = await postgresClient.query(allDevicesQuery);
      deviceIds = allDevicesResult.rows.map(row => row.device_name);
    } else {
      // Get device_ids for the specific user
      const userQuery = `SELECT device_id FROM users WHERE email = $1`;
      const userResult = await postgresClient.query(userQuery, [email]);
      if (userResult.rows.length === 0) {
        return NextResponse.json({ error: 'User not found or no devices assigned' }, { status: 404 });
      }
      deviceIds = userResult.rows.map(row => row.device_id).flat().filter(Boolean);
    }

    if (deviceIds.length === 0) {
      return NextResponse.json({ error: 'No devices found' }, { status: 404 });
    }

    // Query device_max_value for threshold flags
    const flagsQuery = `
      SELECT device_name, "80%", "85%", "90%", "100%" FROM device_max_value WHERE device_name = ANY($1)
    `;
    const flagsResult = await postgresClient.query(flagsQuery, [deviceIds]);

    // Build notifications
    const notifications: { device_id: string; thresholdKey: string; message: string; timestamp: string }[] = [];
    for (const row of flagsResult.rows) {
      ["80%", "85%", "90%", "100%"].forEach(thresholdKey => {
        if (row[thresholdKey] === true) {
          notifications.push({
            device_id: row.device_name,
            thresholdKey,
            message: `Device ${row.device_name} reached ${thresholdKey} usage threshold.`,
            timestamp: new Date().toISOString() // Convert to ISO string
          });
        }
      });
    }

    return NextResponse.json({ notifications });
  } catch (error) {
    console.error('Error fetching device threshold flags:', error);
    return NextResponse.json({ error: 'Failed to fetch device threshold flags', details: error instanceof Error ? error.message : 'Unknown error' }, { status: 500 });
  }
} 