import { NextResponse } from 'next/server';
import { postgresClient } from '../../../../lib/postgresql-client';

export async function POST(request: Request) {
  try {
    const { deviceName } = await request.json();
    if (!deviceName) {
      return NextResponse.json({ error: 'Device name is required' }, { status: 400 });
    }

    // Fetch all columns for the device from device_history
    const query = `SELECT * FROM device_history WHERE device_name = $1`;
    const result = await postgresClient.query(query, [deviceName]);
    if (result.rows.length === 0) {
      return NextResponse.json({ error: 'No history found for device' }, { status: 404 });
    }

    const row = result.rows[0];
    // Remove device_name from the row
    const { device_name, ...historyData } = row;

    // Convert and sort
    const usageArray = Object.entries(historyData)
      .filter(([month, value]) => /^(0[1-9]|1[0-2])_\d{4}$/.test(month))
      .map(([month, value]) => ({
        month,
        usage: value !== null ? Number(value) / 1000 : 0 // liters to m³
      }))
      .sort((a, b) => {
        // Sort by yyyy, then mm
        const [am, ay] = a.month.split('_').map(Number);
        const [bm, by] = b.month.split('_').map(Number);
        return ay !== by ? ay - by : am - bm;
      });

    return NextResponse.json({ deviceName, usage: usageArray });
  } catch (error) {
    console.error('Error fetching device history:', error);
    return NextResponse.json({ error: 'Failed to fetch device history', details: error instanceof Error ? error.message : 'Unknown error' }, { status: 500 });
  }
} 